package ch.sahits.game.openpatrician.model.impl;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IBalanceSheet;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import lombok.Getter;
import org.springframework.beans.factory.annotation.Autowired;
/**
 * 
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Dec 9, 2013
 *
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class BalanceSheet implements IBalanceSheet {

	//	private final Calendar start;
	private final ICity city;
	private final IPlayer player;
	@Autowired
	private Date gameDate;

    @Getter
	private int rentalIncome;
	private int[] rentalIncomeDays;
    @Getter
	private int wageCosts;
	private int[] wageCostsDays;
    @Getter
	private int propertyTaxes;
    @Getter
	private int officeTrading;
	private int[] officeTradingDays;
    @Getter
	private int stewardCost;
	private int[] stewardCostsDays;
    @Getter
	private int otherCosts;
	private int[] otherCostsDays;


	/**
	 * Initialize a blank sheet
	 */
	public BalanceSheet(ICity city, IPlayer player) { 
        super();
		this.city = city;
		this.player = player;

		rentalIncome = 0;
		rentalIncomeDays = new int[]{0,0,0,0,0,0,0};
		wageCosts = 0;
		wageCostsDays = new int[]{0,0,0,0,0,0,0};
		propertyTaxes = 0;
		officeTrading = 0;
		officeTradingDays = new int[]{0,0,0,0,0,0,0};
		stewardCost = 0;
		stewardCostsDays = new int[]{0,0,0,0,0,0,0};
		otherCosts = 0;
		otherCostsDays = new int[]{0,0,0,0,0,0,0};
	}
	/**
	 * Initialize a forcast balance sheet by using the balance sheet of the previous week
	 * @param prevWeek
	 */
	public BalanceSheet(BalanceSheet prevWeek){
		city = prevWeek.city;
		player = prevWeek.player;
		rentalIncome = prevWeek.rentalIncomeDays[6]*7;
		rentalIncomeDays = new int[]{prevWeek.rentalIncomeDays[6],prevWeek.rentalIncomeDays[6],prevWeek.rentalIncomeDays[6],prevWeek.rentalIncomeDays[6],prevWeek.rentalIncomeDays[6],prevWeek.rentalIncomeDays[6],prevWeek.rentalIncomeDays[6]};
		wageCosts = prevWeek.wageCostsDays[6]*7;
		wageCostsDays = new int[]{ prevWeek.wageCostsDays[6], prevWeek.wageCostsDays[6], prevWeek.wageCostsDays[6], prevWeek.wageCostsDays[6], prevWeek.wageCostsDays[6], prevWeek.wageCostsDays[6], prevWeek.wageCostsDays[6]};
        officeTradingDays = new int[]{0,0,0,0,0,0,0};
		officeTradingDays = new int[]{prevWeek.officeTradingDays[6],prevWeek.officeTradingDays[6],prevWeek.officeTradingDays[6],prevWeek.officeTradingDays[6],prevWeek.officeTradingDays[6],prevWeek.officeTradingDays[6],prevWeek.officeTradingDays[6]};
		stewardCost = prevWeek.stewardCostsDays[6]*7;
		stewardCostsDays = new int[]{prevWeek.stewardCostsDays[6],prevWeek.stewardCostsDays[6],prevWeek.stewardCostsDays[6],prevWeek.stewardCostsDays[6],prevWeek.stewardCostsDays[6],prevWeek.stewardCostsDays[6],prevWeek.stewardCostsDays[6]};
		otherCosts = prevWeek.otherCosts;
		otherCostsDays = new int[]{prevWeek.otherCosts/7,prevWeek.otherCosts/7,prevWeek.otherCosts/7,prevWeek.otherCosts/7,prevWeek.otherCosts/7,prevWeek.otherCosts/7,prevWeek.otherCosts/7};
	}

	/**
	 * Update the costs for the steward.
	 * @param costPerDay daily salary costs of the stewards in all the trading offices.
     */
	@Override
	public void updateStewardCosts(int costPerDay) {
		int weekIndex = getWeekIndex();
		for (int i = weekIndex; i<stewardCostsDays.length;i++){
			stewardCostsDays[i] = costPerDay;
		}
		int sum = 0;
		for (int cost : stewardCostsDays) {
			sum += cost;
		}
		stewardCost=sum;
	}

	/**
	 * Update the income for rented houses.
	 * @param incomePerDay amount payed per day
     */
	@Override
	public void updateRentalIncome(int incomePerDay) {
		int weekIndex = getWeekIndex();
		for (int i = weekIndex; i<rentalIncomeDays.length;i++){
			rentalIncomeDays[i] = incomePerDay;
		}
		int sum = 0;
		for (int cost : rentalIncomeDays) {
			sum += cost;
		}
		rentalIncome=sum;
	}

	/**
	 * Update the property taxes for the whole week.
	 * @param amount property taxes for the week
     */
	@Override
	public void deductPropertyTaxes(int amount) {
		propertyTaxes = amount;
	}

	/**
	 * Update the saleries per day.
	 * @param saleriesPerDay
     */
	@Override
	public void updateSalaries(int saleriesPerDay) {
		int weekIndex = getWeekIndex();
		for (int i = weekIndex; i<wageCostsDays.length;i++){
			wageCostsDays[i] = saleriesPerDay;
		}
		int sum = 0;
		for (int cost : wageCostsDays) {
			sum += cost;
		}
		wageCosts=sum;
	}

	/**
	 * Update the costs incurred by automatic trading for the day
	 * @param costs
     */
	@Override
	public void updateAutomatedTradingCosts(long costs) {
        int weekIndex = getWeekIndex();
        officeTradingDays[weekIndex - 1] = (int)costs;
        officeTrading += costs;
	}

	/**
	 * Update any regular other costs.
	 * @param costsPerDay
     */
	@Override
	public void updateOtherExpensesRegular(int costsPerDay) {
		int weekIndex = getWeekIndex();
		for (int i = weekIndex; i<otherCostsDays.length;i++){
			otherCostsDays[i] = costsPerDay;
		}
		int sum = 0;
		for (int cost : otherCostsDays) {
			sum += cost;
		}
		otherCosts=sum;
	}

	/**
	 * Update the other costs with a one time expense.
	 * @param costs
     */
	@Override
	public void updateOtherExpensesOneTime(int costs) {
		int weekIndex = getWeekIndex();
		otherCostsDays[weekIndex - 1] = costs;
		otherCosts += costs;
	}

	private int getWeekIndex() {
		return gameDate.getCurrentDate().getDayOfWeek().getValue();
	}

}
