package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DateService;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.OptionalType;
import ch.sahits.game.openpatrician.utilities.service.RandomNameLoader;
import com.google.common.base.Preconditions;
import com.google.common.collect.Range;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Random;

/**
 * Implementation of the captain.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 26, 2013
 *
 */
@Component
@Scope("prototype")
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class CaptainState implements ICaptain {
	@Autowired
	private Date date;
	@Autowired
	@XStreamOmitField
	private DateService dateService;
	@Autowired
	@XStreamOmitField
	private Random rnd;
	private LocalDateTime birthDate;
	private String name;

	private int salary;
	private int tradingSkill;
	private int navigationSkill;
	private int fightingSkill;
	private int totalProfit;
	private long sailedDistance;
	private int wonFights;
	@OptionalType(IShip.class)
	private Optional<IShip> ship;

	private final static Range<Integer> skillRange = Range.closed(0, 5);

	/** Limits for the level that have to be reached before upgrading navigation skill. */
	@MapType(key = Integer.class, value = Integer.class)
	private final static Map<Integer, Integer> UPGRADE_NAVIGATION_LIMIT_LEVEL = new HashMap<>();
	/** Limits for the level that have to be reached before upgrading fighting skill. */
	@MapType(key = Integer.class, value = Integer.class)
	private final static Map<Integer, Integer> UPGRADE_FIGHT_LIMIT_LEVEL = new HashMap<>();
	/** Limits for the level that have to be reached before upgrading. */
	@MapType(key = Integer.class, value = Integer.class)
	private final static Map<Integer, Integer> UPGRADE_TRADE_LIMIT_LEVEL = new HashMap<>();

	{
		UPGRADE_TRADE_LIMIT_LEVEL.put(0, 10000);
		UPGRADE_TRADE_LIMIT_LEVEL.put(1, 20000);
		UPGRADE_TRADE_LIMIT_LEVEL.put(2, 40000);
		UPGRADE_TRADE_LIMIT_LEVEL.put(3, 70000);
		UPGRADE_TRADE_LIMIT_LEVEL.put(4, 100000);
		UPGRADE_NAVIGATION_LIMIT_LEVEL.put(0, 10000);
		UPGRADE_NAVIGATION_LIMIT_LEVEL.put(1, 20000);
		UPGRADE_NAVIGATION_LIMIT_LEVEL.put(2, 40000);
		UPGRADE_NAVIGATION_LIMIT_LEVEL.put(3, 70000);
		UPGRADE_NAVIGATION_LIMIT_LEVEL.put(4, 100000);
		UPGRADE_FIGHT_LIMIT_LEVEL.put(0, 5);
		UPGRADE_FIGHT_LIMIT_LEVEL.put(1, 10);
		UPGRADE_FIGHT_LIMIT_LEVEL.put(2, 15);
		UPGRADE_FIGHT_LIMIT_LEVEL.put(3, 25);
		UPGRADE_FIGHT_LIMIT_LEVEL.put(4, 40);
	}

	private static RandomNameLoader firstNameLoader = new RandomNameLoader("captain-firstnames.properties");
	private static RandomNameLoader lastNameLoader = new RandomNameLoader("captain-lastnames.properties");

	@PostConstruct
	private void init() {
		name = firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName();
		birthDate = date.getCurrentDate().minusYears(rnd.nextInt(15)+25)
				.minusMonths(rnd.nextInt(12))
				.minusDays(rnd.nextInt(30));
		setFightSkillLevel(rnd.nextInt(6));
		setTradingSkillLevel(rnd.nextInt(6));
		setNavigationSkillLevel(rnd.nextInt(6));
		int age = getAge();
		int experianceLevel = (int)Math.min(5,rnd.nextInt(6) * (age/50.0));
		setSalary(experianceLevel+10+tradingSkill*2+navigationSkill*2+fightingSkill*2);
		ship = Optional.empty();
		totalProfit = 0;
	}


	@Override
	public int getAge() {
		return dateService.getAge(birthDate);
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public int getSalary() {
		return salary;
	}

	@Override
	public int getTradingSkillLevel() {
		return tradingSkill;
	}

	@Override
	public int getNavigationSkillLevel() {
		return navigationSkill;
	}

	@Override
	public int getFightSkillLevel() {
		return fightingSkill;
	}

	/**
	 * Set the new trading skill level between [0..5]
	 * @param level
	 */
	private void setTradingSkillLevel(int level) {
		Preconditions.checkArgument(skillRange.contains(level), "Level must be within [0,5]");
		tradingSkill = level;
	}

	/**
	 * Set the new navigation skill level between [0..5]
	 * @param level
	 */
	private void setNavigationSkillLevel(int level) {
		Preconditions.checkArgument(skillRange.contains(level), "Level must be within [0,5]");
		navigationSkill = level;
	}

	/**
	 * Set the new fight skill level between [0..5]
	 * @param level
	 */
	private void setFightSkillLevel(int level) {
		Preconditions.checkArgument(skillRange.contains(level), "Level must be within [0,5]");
		fightingSkill = level;
	}

	/**
	 * Set the new salary
	 * @param salary
	 */
	public void setSalary(int salary) {
		this.salary = salary;
	}

	public void assignToShip(IShip ship) {
		ship.setCaptain(this);
	}

	/**
	 * Get the ship the captain is assigned to.
	 * @return absent if the captain is not assigned to any ship.
     */
	public Optional<IShip> getAssignedShip() {
		return ship;
	}


	@Override
	public boolean upgradeToNextTradeLevel() {
		if (tradingSkill < 5 && totalProfit > UPGRADE_TRADE_LIMIT_LEVEL.get(navigationSkill)) {
			tradingSkill++;
			salary = (int) (1.8 * salary);
			return true;
		}
		return false;
	}

	@Override
	public void updateProfit(int profit) {
		totalProfit += profit;
	}

	@Override
	public void updatedSailedDistance(int distance) {
		sailedDistance += distance;
	}

	@Override
	public boolean upgradeToNextNavigationLevel() {
		if (navigationSkill < 5 && sailedDistance > UPGRADE_NAVIGATION_LIMIT_LEVEL.get(navigationSkill)) {
			navigationSkill++;
			salary = (int) (1.8 * salary);
			return true;
		}
		return false;
	}

	@Override
	public void updateFightWon() {
		wonFights++;
	}

	@Override
	public boolean upgradeToNextFightingLevel() {
		if (fightingSkill < 5 && wonFights > UPGRADE_FIGHT_LIMIT_LEVEL.get(fightingSkill)) {
			fightingSkill++;
			salary = (int) (1.8 * salary);
			return true;
		}
		return false;
	}
}
