package ch.sahits.game.openpatrician.model.player;

import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.IShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;

/**
 * This interface defines the API for a trading strategy for the AI.
 */
public interface IAITradeStrategy {
    /**
     * Initialize the trade cycle. Based on the strategy the cycle may be closed,
     * meaning at some point this method can be called again or it may be open
     * with destination defining the next one.
     * @param player
     * @param vessel
     */
    void initializeTradeCycle(IAIPlayer player, INavigableVessel vessel);

    /**
     * In the case where the trade cycle must be re-initialized, but the ship is not in a city,
     * determine the city to which the vessel should travel.
     * @param vessel that should travel to a city, so that the trade cycle can be reinitialized
     * @return city to which the vessel travels.
     */
    ICity getCityToRestartTradeCycle(INavigableVessel vessel);

    /**
     * Execute the trad steps defined for the vessel.
     * @param player
     * @param vessel
     */
    boolean executeTradeSteps(IAIPlayer player, INavigableVessel vessel);

    /**
     * Event handler for a ship entering port.
     * @param event
     */
    void handleShipArrivesInPort(IShipEntersPortEvent event);

    /**
     * Check if this strategy can be selected in the current setup.
     * This method is called when choosing a new strategy for the <code>vessel</code>.
     * @param player for which the strategy is chosen.
     * @param vessel for which the strategy is chose.
     * @return true if the strategy can be chosen for the <code>player</code> and <code>vessel</code>. The
     * default implementation will always return true.
     */
    default boolean isSelectable(IAIPlayer player, INavigableVessel vessel) {
        return true;
    }
}
