package ch.sahits.game.openpatrician.model.service;

import ch.sahits.game.openpatrician.model.IModelTranslationService;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.model.personal.EEconomicCareer;
import ch.sahits.game.openpatrician.model.personal.EMilitantCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;

/**
 * Service that provides the proper translations for enumeration values and constants.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Nov 26, 2014
 */
@Service
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class ModelTranslations implements IModelTranslationService {
    @Autowired
    private Locale locale;
    @Autowired
    @Qualifier("messageSource")
    private MessageSource messages;

    /**
     * Retrieve the localized display name for <code>id</code>.
     * @param weapon ennumeration
     * @return
     */
    public String getLocalDisplayName(EWeapon weapon) {
        return messages.getMessage(weapon.name(), new Object[0], locale.getCurrentLocal());
    }
    /**
     * Retrieve the localized display name for <code>id</code>.
     * @param ware
     * @return
     */
    public String getLocalDisplayName(EWare ware){
        return messages.getMessage(ware.getWareID(), new Object[0], locale.getCurrentLocal());
    }

    public String getLocalDisplayName(ICareer career) {
        if (career instanceof EEconomicCareer) {
            return messages.getMessage(((EEconomicCareer)career).name(), new Object[0], locale.getCurrentLocal());
        } else if (career instanceof EMilitantCareer) {
            return messages.getMessage(((EMilitantCareer)career).name(), new Object[0], locale.getCurrentLocal());
        } else {
            throw new IllegalArgumentException("Unknown career implementation: "+career.getClass().getName());
        }
    }

    public String getLocalDisplayName(ESocialRank rank) {
        return messages.getMessage(rank.name(), new Object[0], locale.getCurrentLocal());
    }
    public String getLocalDisplayName(EShipType type) {
        return messages.getMessage(type.name(), new Object[0], locale.getCurrentLocal());
    }
    public String getLocalDisplayNameWithArticle(EShipType type, boolean upperCaseArticle) {
        String typeName = getLocalDisplayName(type);
        String article;
        if (EShipType.HOLK == type || EShipType.CRAYER == type) {
            article = getLocalDisplayDistinctArticle(true, upperCaseArticle);
        } else {
            article = getLocalDisplayDistinctArticle(false, upperCaseArticle);
        }
        return article + " " + typeName;
    }

    public String getLocalDisplayDistinctArticle(boolean male, boolean uppercase) {
        String key = "articleMale";
        if (!male) {
            key = "articleFemale";
        }
        String article = messages.getMessage(key, new Object[0], locale.getCurrentLocal());
        if (uppercase) {
            article = article.substring(0,1).toUpperCase() + article.substring(1);
        }
        return article;
    }
    /**
     * Convert the date into the form {Day in month}. {Month}
     * @param c
     * @return
     */
    public String toShortDate(LocalDateTime c) {
        final int day = c.getDayOfMonth();
        final int month = c.getMonthValue();
        StringBuilder sb = new StringBuilder();
        sb.append(day).append(". ");
        sb.append(messages.getMessage("month_short_"+month, new Object[]{}, locale.getCurrentLocal()));

        return sb.toString();
    }
    /**
     * Retrieve the date of the of the Sunday of last week
     * @return
     */
    private LocalDateTime getEndOfWeekInternal(LocalDateTime date) {
        int cdow = date.getDayOfWeek().getValue();
        LocalDateTime lastDayOfWeek = date.plusDays(7 - cdow);
        return lastDayOfWeek;
    }
    /**
     * Retrieve the date of the end of the week in the form {Day of month}. {Month}.
     * The week ends with Sunday
     * @return
     */
    public String getEndOfWeek(LocalDateTime date){
        LocalDateTime lastSun = getEndOfWeekInternal(date);
        return toShortDate(lastSun);
    }
    public String toDisplayString(final int day, final int month, final int year) {
        StringBuilder sb = new StringBuilder();
        sb.append(day).append(". ");
        sb.append(messages.getMessage("month_short_" + month, new Object[]{}, locale.getCurrentLocal())).append(" ");
        sb.append(year);
        return sb.toString();
    }
    private String toDisplayStringLongFormat(final int day, final int month, final int year) {
        StringBuilder sb = new StringBuilder();
        sb.append(day).append(". ");
        sb.append(messages.getMessage("month_long_" + month, new Object[]{}, locale.getCurrentLocal())).append(" ");
        sb.append(year);
        return sb.toString();
    }
    /**
     * Create a string representation for the user interface of the
     * form {Day of month}. {Month} {Year}.
     * @return
     */
    public String toDisplayString(LocalDateTime date){
        final int day = date.getDayOfMonth();
        final int month = date.getMonthValue();
        final int year = date.getYear();
        return toDisplayString(day, month, year);
    }

    /**
     * Create a string representation for the user interface of the
     * form {Day of month}. {Month} {Year} with long month format.
     * @return
     */
    public String toDisplayStringLongFormat(LocalDateTime date){
        final int day = date.getDayOfMonth();
        final int month = date.getMonthValue();
        final int year = date.getYear();
        return toDisplayStringLongFormat(day, month, year);
    }
    /**
     * Retrieve the date of the start of the week in the form {Day of month}. {Month}.
     * The week starts with Monday
     * @return
     */
    public String getStartOfWeek(LocalDateTime date){
        LocalDateTime lastMon = getStartOfWeekInternal(date);
        return toShortDate(lastMon);
    }
    /**
     * Retrieve the date of the start of the last week
     * @return
     */
    private LocalDateTime getStartOfWeekInternal(LocalDateTime date) {
        // first day
        LocalDateTime lastMonday = date.minusDays(date.getDayOfWeek().getValue()-1);
        return lastMonday;
    }

    public String getLocalDisplayName(IBuilding building) {
        return messages.getMessage("building.type"+building.getClass().getSimpleName(), new Object[]{}, locale.getCurrentLocal());
    }
}
