package ch.sahits.game.openpatrician.model.ship.impl;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MultimapType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.model.javafx.bindings.LateIntegerBinding;
import ch.sahits.game.openpatrician.model.AmountableProvider;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.ICog;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.ICrayer;
import ch.sahits.game.openpatrician.model.ship.IHolk;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.ISnaikka;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.geometry.Point2D;
import lombok.Getter;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;

/**
 * Implementation of the Convoy.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 23, 2015
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class Convoy implements IConvoy {
    @Autowired
    @XStreamOmitField
    private Random rnd;
    @Autowired
    @XStreamOmitField
    private AmountableProvider amountableProvider;
    private final IShip orlegShip;
    @Getter
    private final boolean publicConvoy;
    @MultimapType(key = IPlayer.class, value = IShip.class)
    private Multimap<IPlayer, IShip> ships = ArrayListMultimap.create();
    /**
     * Binding representing the current load.
     */
    @XStreamOmitField
    private LateIntegerBinding loadBinding = null;

    public Convoy(IShip orlegShip, boolean publicConvoy) {
        this.orlegShip = orlegShip;
        this.publicConvoy = publicConvoy;
        ships.put((IPlayer) orlegShip.getOwner(), orlegShip);
    }

    @Override
    public boolean hasWeapons() {
        return false;
    }

    @Override
    public IShip getOrlegShip() {
        return orlegShip;
    }

    @Override
    public List<IShip> getShips() {
        ArrayList<IShip> allShips = new ArrayList<>(ships.values());
        return allShips;
    }

    private LateIntegerBinding loadbinding() {
        if (loadBinding == null) {
            loadBinding = createLoadBinding();
        }
        return loadBinding;
    }

    @Override
    public void addShip(IShip ship) {
        ships.put((IPlayer) ship.getOwner(), ship);
        loadbinding().bind(ship.getLoadBinding());
        loadbinding().invalidate();
    }

    @Override
    public void removeShip(IShip ship) {
       ships.remove(ship.getOwner(), ship);
        loadbinding().unbind(ship.getLoadBinding());
        loadbinding().invalidate();

    }
    private LateIntegerBinding createLoadBinding() {
        return new LateIntegerBinding() {
            {
                for (IShip ship : ships.values()) {
                    super.bind(ship.getLoadBinding());
                }
            }

            @Override
            protected int computeValue() {
                return ships.values().stream().mapToInt(ship -> ship.getLoadBinding().get()).sum();
            }

        };
    }

    @Override
    public int getSize() {
        return ships.values().stream().mapToInt(ship -> ship.getSize()).sum();
    }

    @Override
    public int getCapacity() {
        return ships.values().stream().mapToInt(ship -> ship.getCapacity()).sum();

    }


    /**
     * The name of the convoy is the name of the Orleg ship.
     * @return
     */
    @Override
    public String getName() {
        return orlegShip.getName();
    }

    @Override
    public Set<IWare> getLoadedWares() {
        Set<IWare> loadedWares = new HashSet<>();
        for (IShip ship : ships.values()) {
            loadedWares.addAll(ship.getLoadedWares());
        }
        return loadedWares;
    }

    /**
     * The wares are loaded onto the ships in the convoy
     * in the iteration order. There is no guarantee which
     * ware is put on which ship.
     * @return amount that is loaded.
     */
    @Override
    public int load(IWare ware, int amount, int avgPrice) {
        int loadedAmount = 0;
        int remainingAmount = amount;
        for (IShip ship : ships.values()) {
            if (ship.getCapacity() > 0) {
               loadedAmount += ship.load(ware, remainingAmount, avgPrice);
                remainingAmount = amount - loadedAmount;
                if (remainingAmount <= 0) {
                    break;
                }
            }
        }
        return loadedAmount;
    }

    /**
     * Unload the wares from the ships in the convoy in the iteration
     * order. There is no guarantee from which ship the ware will be unloaded
     * if thot the compleate loaded amount is unloaded.
     * @param ware to be unloaded
     * @param amount amount of items of ware to unload
     * @return amount of the ware that is actually unloaded.
     */
    @Override
    public int unload(IWare ware, int amount) {
        int unloadedAmount = 0;
        int remainingAmount = amount;
        for (IShip ship : ships.values()) {
            unloadedAmount += ship.unload(ware, remainingAmount);
            remainingAmount = amount - unloadedAmount;
            if (remainingAmount <= 0) {
                break;
            }
        }
        return unloadedAmount;
    }

    /**
     * Retrieve the damage of the ship that is damaged the most.
     * @return damage of the most damaged ship in range [100,0]
     */
    @Override
    public int getDamage() {
        return ships.values().stream().mapToInt(ship -> ship.getDamage()).max().getAsInt();
    }

    /**
     * Apply the damage to all ships. the actual damage varies based on ship type and
     * some random factor.
     * @param damage average damage by ship
     */
    @Override
    public void damage(int damage, boolean destroyWeapon) {
        for (IShip ship : ships.values()) {
            double rndFactor = Math.max(rnd.nextGaussian() + 1, 0);
            double typeFactor = 1;
            if (ship instanceof IHolk) {
                typeFactor = 0.75;
            }
            if (ship instanceof ICog) {
                typeFactor = 1;
            }
            if (ship instanceof ICrayer) {
                typeFactor = 1.3;
            }
            if (ship instanceof ISnaikka) {
                typeFactor = 1.7;
            }
            int actualDamage = (int) Math.round(damage*rndFactor*typeFactor);
            ship.damage(actualDamage, destroyWeapon);
        }

    }


    @Override
    public IntegerBinding getLoadBinding() {
        return loadbinding();
    }

    @Override
    public AmountablePrice<IWare> getWare(IWare ware) {
        AmountablePrice<IWare> loadedWare = amountableProvider.createWareAmountable();
        for (IShip ship : ships.values()) {
            AmountablePrice<IWare> amountable = ship.getWare(ware);
            loadedWare.add(amountable.getAmount(), amountable.getAVGPrice());
        }
        return loadedWare;
    }


    @Override
    public int getNumberOfSailors() {
        return ships.values().stream().mapToInt(ship -> ship.getNumberOfSailors()).sum();
    }

    /**
     * Retrieve the captain of the Orleg ship.
     * @return
     */
    @Override
    public Optional<ICaptain> getCaptian() {
        return orlegShip.getCaptian();
    }


    @Override
    public int getWeaponAmount(IWeapon weaponType) {
        return ships.values().stream().mapToInt(ship -> ship.getWeaponAmount(weaponType)).sum();
    }


    @Override
    public Point2D getLocation() {
        return orlegShip.getLocation();
    }

    @Override
    public void setLocation(Point2D location) {
        for (IShip ship : ships.values()) {
            ship.setLocation(location);
        }
    }

    @Override
    public IShipOwner getOwner() {
        return orlegShip.getOwner();
    }


    @Override
    public List<IShip> getShips(IPlayer player) {
        return new ArrayList<>(ships.get(player));
    }

    @Override
    public List<IPlayer> getPlayers() {
        return new ArrayList<>(ships.keySet());
    }

    @Override
    public Map<IPlayer, Integer> getCapacityPerOwner() {
        Map<IPlayer, Integer> map = new HashMap<>();
        for (IPlayer player : ships.keys()) {
            int capacity = getShips(player).stream().mapToInt(ship -> ship.getCapacity()).sum();
            map.put(player, capacity);
        }
        return map;
    }

    @Override
    public int getLoadableSpace() {
        return ships.values().stream().mapToInt(ship -> ship.getLoadableSpace()).sum();
    }

    @Override
    public double getTopSpeed() {
        return ships.values().stream().mapToDouble(ship -> ship.getTopSpeed()).max().getAsDouble();
    }
    @Override
    public double getCurrentSpeed() {
        return ships.values().stream().mapToDouble(ship -> ship.getCurrentSpeed()).min().getAsDouble();
    }

    @Override
    public String getUuid() {
        return getOrlegShip().getUuid();
    }

    @Override
    public boolean getPirateFlag() {
        return false;
    }

    @Override
    public void togglePirateFlag() {
       // no operation
    }

    @Override
    public BooleanProperty pirateFlagProperty() {
        return new SimpleBooleanProperty(false);
    }
}
