package ch.sahits.game.openpatrician.model;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.javafx.IJavaFXApplicationThreadExecution;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * This base class for amountable items provides a count of the items.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Dec 23, 2011
 *
 * @param <T>
 */
@Prototype
@ClassCategory({EClassCategory.PROTOTYPE_BEAN, EClassCategory.SERIALIZABLE_BEAN})
public class BaseAmountable<T extends IAmountable> {
	/** Number of items that are amounted */
	private final IntegerProperty amount = new SimpleIntegerProperty(this, "amount", 0);
    @Autowired
    @XStreamOmitField
    protected IJavaFXApplicationThreadExecution fxTheadExecution;

    /**
	 * Retrieve the number of items
	 * @return
	 */
	public int getAmount(){
		return amount.get();
	}

	protected void setAmount(int amount) {
		Preconditions.checkArgument(amount >= 0, "The amount must not be negative: "+amount);
		this.amount.set(amount);
	}
	public ReadOnlyIntegerProperty amountProperty() {
		return amount;
	}

	/**
	 * Add a number items
	 * @param amount number of the items to be added
	 */
	public void add(final int amount){
		Preconditions.checkArgument(amount >= 0, "Amount may not be negative: "+amount);
        fxTheadExecution.execute(() -> {
			int localAmount = amount;
			if (localAmount<0){
				localAmount = Math.abs(localAmount);
			}
			this.amount.set(this.getAmount() + localAmount);
		});
	}
	/**
	 * Remove a number of items
	 * @param amount of items to be removed
	 */
	public void remove (final int amount){
		Preconditions.checkArgument(amount >= 0, "Amount may not be negative: "+amount);
        fxTheadExecution.execute(() -> {
			int localAmount = amount;
			if (localAmount<0){
				localAmount = Math.abs(localAmount);
			}
			if (localAmount==Integer.MIN_VALUE){
				localAmount = Integer.MAX_VALUE;
			}
			if (this.amount.get() < localAmount){
				this.amount.set(0);
			} else {
				this.amount.set(this.getAmount() - localAmount);
			}
		});

	}
	/**
	 * Set the amount to 0.
	 */
	public void reset() {
        fxTheadExecution.execute(() -> {
            this.amount.set(0);
        });
	}
	public String toString(){
		return String.valueOf(amount.get());
	}
//	public IntegerProperty amountProperty() {
//		return amount;
//	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		BaseAmountable<?> that = (BaseAmountable<?>) o;

		return amount != null ? amount.get() == that.amount.get() : that.amount == null;

	}

	@Override
	public int hashCode() {
		return amount != null ? amount.hashCode() : 0;
	}
}
