package ch.sahits.game.openpatrician.model;

import ch.sahits.game.openpatrician.model.player.AIPlayerContext;
import ch.sahits.game.openpatrician.model.player.IAIBuyWeaponStrategyType;
import ch.sahits.game.openpatrician.model.player.IAICaptainHireStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIConstructionSelectionStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIEventDecisionStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIGuildJoinStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIHireSailorsStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIHireTradeManagerStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIShipRepairStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIShipUpgradeStrategyType;
import ch.sahits.game.openpatrician.model.player.IAITakeLoanStrategyType;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategyType;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;

/**
 * Player interface defining an artificial player.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 17, 2011
 *
 */
public interface IAIPlayer extends IPlayer {
    void setTradeStrategyType(INavigableVessel vessel, IAITradeStrategyType type);

    /**
     * Retreive the the type of trade strategy.
     * @return
     */
    IAITradeStrategyType getTradeStrategyType(INavigableVessel vessel);

    /**
     * Retrieve the type of for the ship repair strategy.
     * @return
     */
    IAIShipRepairStrategyType getShipRepairStrategyType();

    /**
     * Retrieve the type of the strategy for taking a loan.
     * @return
     */
    IAITakeLoanStrategyType getTakeLoanStrategyType();

    /**
     * Retrieve the strategy type for event decission.
     * @return
     */
    IAIEventDecisionStrategyType getEventDecitionStrategyType();

    /**
     * Retrieve the type of the strategy to use for deciding how to build new ships.
     * @return
     */
    IAIConstructionSelectionStrategyType getConstructionSelectionType();

    /**
     * Retrieve the type of the strategy to hire captains.
     * @return
     */
    IAICaptainHireStrategyType getCaptainHireStrategyType();

    /**
     * Retrieve the strategy to join guilds.
     * @return
     */
    IAIGuildJoinStrategyType getGuildJoinStrategyType();

    /**
     * Retrieve the strategy type for buying weapons.
     * @return
     */
    IAIBuyWeaponStrategyType getWeaponBuyStrategyType();

    /**
     * Retrieve the strategy type for upgrading ships.
     * @return
     */
    IAIShipUpgradeStrategyType getShipUpgradeStrategyType();

    /**
     * Retreive the strategy type for hiring the sailors.
     * @return
     */
    IAIHireSailorsStrategyType getHireSailorStrategyType();

    /**
     * Retrieve the strategy for hiring and dismissing trade managers.
     * @return
     */
    IAIHireTradeManagerStrategyType getHireTradeManagerStrategyType();

    /**
     * Retrieve the knowledge of production and consumption.
     * @return
     */
    IProductionConsumptionKnowledge getProductionAndConsumptionKnowledge();

    /**
     * Retrieve the next trade step that should be executed and remove it from the list.
     * @param vessel for which the next trade step should be checked.
     * @return
     */
    ITradeStep getNextTradeStep(INavigableVessel vessel);

    /**
     * Check if there are further trade steps.
     * @param vessel which should be checked.
     * @return
     */
    boolean hasMoreTradeSteps(INavigableVessel vessel);

    /**
     * Add a new trade step at the end of the list.
     * @param vessel for which the step should be added.
     * @param tradeStep to be added.
     */
    void addTradeStep(ITradeStep tradeStep, INavigableVessel vessel);

    /**
     * Retrieve the trade mission of a given vessel.
     * @param vessel
     * @return
     */
    ITradeMissionData getTradeMission(INavigableVessel vessel);

    /**
     * Add trade mission data fo a vessel. If the data is null an existing entry will be removed.
     * @param vessel
     * @param tradeMission
     */
    void setTradeMission(INavigableVessel vessel, ITradeMissionData tradeMission);

    /**
     * Inject a trade step at the beinning of the task list for the vessel
     * @param vessel for which the step should be added.
     * @param tradeStep to be added.
     */
    void injectTradeStep(ITradeStep tradeStep, INavigableVessel vessel);

    /**
     * Check if the next trade step can be executed for the vessel, or if the last
     * trade step requires waiting on an event.
     * @param vessel for which the status should be checked.
     * @return true if the next trade step is initiated upon an event.
     */
    boolean waitingForTradeStepToFinish(INavigableVessel vessel);

    /**
     * Update  the waiting status upon the execution of a trade step.
     * @param vessel for which the status should be updated.
     * @param wait inicate if the next trade step should wait.
     */
    void updateTradeWaitingStatus(INavigableVessel vessel, boolean wait);

    /**
     * Check if the trade strategy for the vessel is initialized.
     * @param vessel to be checked
     * @return true if the trade steps for the vessel are initialized.
     */
    boolean isInitialized(INavigableVessel vessel);

    /**
     * Retrieve the context for this player.
     * @return
     */
    AIPlayerContext getPlayerContext();

}
