package ch.sahits.game.openpatrician.model;

import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.map.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.personal.EEconomicCareer;
import ch.sahits.game.openpatrician.model.personal.EMilitantCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import ch.sahits.game.openpatrician.model.personal.IChild;
import ch.sahits.game.openpatrician.model.personal.IPersonalData;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;

import java.util.List;
import java.util.Optional;

/**
 * Base player in the game.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Jun 15, 2011
 *
 */
public interface IPlayer extends ICitizen, IShipOwner{
	/**
	 * Retrieve the personal data of the player
	 * @return
	 */
	IPersonalData getPersonalData();
	/**
	 * Retrieve the personal data of the players spouse.
	 * @return
	 */
	Optional<ISpouseData> getSpouseData();
	/**
	 * Retrieve the company data
	 * @return
	 */
	ICompany getCompany();

	/**
	 * Get a list of all ships
	 * @return
	 */
	List<IShip> getFleet();

	/**
	 * Provide a different view on the fleet by containing all the
	 * vessels that can be selected.
	 * @return list of vessels that can be activated.
     */
	List<INavigableVessel> getSelectableVessels();

    /**
     * Add a selectable vessel to the list.
     * @param vessel to be added
     */
    void addSelectableVessel(INavigableVessel vessel);

    /**
     * Remove a selectable vessel from the list.
     * @param vessel to be removed.
     */
    void removeSelectableVessel(INavigableVessel vessel);
	/**
	 * Find all ships in a city
	 * @param city
	 * @return
	 */
	List<INavigableVessel> findShips(ICity city);
	/**
	 * Add a new ship for the player
	 * @param ship
	 */
	void addShip(IShip ship);
	/**
	 * Remove a ship for the player
	 * @param ship
	 */
	void removeShip(IShip ship);
	/**
	 * Find all buildings of the player in the city
	 * @param city
	 * @return
	 */
	List<IBuilding> findBuildings(ICity city);
	/**
	 * Find the trading office of the player in the city.
	 * @param city
	 * @return
	 */
	Optional<ITradingOffice> findTradingOffice(ICity city);
	/**
	 * Find all buildings in the city that are some subtype of the indicated class
	 * @param city
	 * @param buildingClass
	 * @return
	 */
	<T extends IBuilding> List<T> findBuildings(ICity city, Class<T> buildingClass);

	/**
	 * Update to the new social rank
	 * @param rank
	 */
	void updateRank(ESocialRank rank);
	/**
	 * Retrieve the career level, which is either defined by {@link EMilitantCareer} or {@link EEconomicCareer}.
	 * @return
	 */
	ICareer getCareerLevel();

    /**
     * Set the current career level.
     * @param career
     */
	void setCareerLevel(ICareer career);

	/**
	 * Set or unset the image for a segmented map (treasure or pirate hideout).
	 * @param mapSegments
	 */
	void setSegmentedMap(MapSegmentedImage mapSegments);

	/**
	 * Retrieve the segmented image map.
	 * @return
	 */
	MapSegmentedImage getSegmentedMap();

	/**
	 * Conclude the marriage.
	 * @param spouse
	 */
	void marry(ISpouseData spouse);

	/**
     * Widow and become unmarried again.
	 */
	void spouseDies();

    /**
     * Retrieve all children.
     * @return
     */
    List<IChild> getChildren();

	/**
	 * Update the new value indicating the level of criminallity of the player
	 * @param value
	 */
	void updateCrimialDrive(int value);

	/**
	 * Retrieve the value indicating the criminal drive of the player.
	 * @return
	 */
	int getCriminalDrive();

	/**
	 * Retrieve the UUID identifing this player.
 	 * @return
	 */
	String getUuid();
}
