package ch.sahits.game.openpatrician.model;

import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.city.guild.IBuildingAuction;
import ch.sahits.game.openpatrician.model.city.guild.IShipAuction;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.time.LocalDateTime;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Nov 13, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class ModelFactory {
    @Autowired
    private ApplicationContext context;

    /**
     * Retrieve an auction instance for the <code>building</code> owned by <code>owner</code>, which
     * will be auctioned on <code>auctionDate</code> starting with <code>startingBid</code>
     * @param auctionDate date of the auction
     * @param startingBid first bid amount
     * @param owner of the building to be auctioned.
     * @param building that is auctioned.
     * @return
     */
    public IBuildingAuction createBuildingAuction(LocalDateTime auctionDate, int startingBid, IPlayer owner, IBuilding building) {
        return (IBuildingAuction) context.getBean("buildingAuction", auctionDate, startingBid, owner, building);
    }

    /**
     * Retrieve an auction instance for a <code>ship</code>, that has no previous owner, e.g. because it
     * was captured during a blockade mission. The auction will happen on <code>auctionDate</code> and
     * start with <code>startingBid</code>.
     * @param auctionDate date of the auction
     * @param startingBid first bid amount
     * @param ship that is auctioned.
     * @return
     */
    public IShipAuction createAnonymousShipAuction(LocalDateTime auctionDate, int startingBid, IShip ship){
        return (IShipAuction) context.getBean("shipAuction", auctionDate, startingBid, ship);
    }

    /**
     * Retrieve an auction instance for the <code>ship</code> owned by <code>owner</code>, which
     * will be auctioned on <code>auctionDate</code> starting with <code>startingBid</code>
     * @param auctionDate date of the auction
     * @param startingBid first bid amount
     * @param owner of the ship that is auctioned
     * @param ship that is auctioned.
     * @return
     */
    public IShipAuction createShipAuction(LocalDateTime auctionDate, int startingBid, IPlayer owner, IShip ship){
        return (IShipAuction) context.getBean("shipAuction", auctionDate, startingBid, owner, ship);
    }
}
