package ch.sahits.game.openpatrician.model.building;

import ch.sahits.game.openpatrician.model.IBalanceSheet;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;

import java.time.LocalDateTime;
import java.util.Optional;

/**
 * The trading office (Kontor) is the trading point in a city. You can buy/sell between the trading
 * office and the city. You can move wares and wares between your storage and any ship currently in
 * the city.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Dec 17, 2011
 *
 */
public interface ITradingOffice extends IBuilding {
	/**
	 * Retrieve the amount of ware in storage
	 * @param ware
	 * @return
	 */
	AmountablePrice<IWare> getWare(IWare ware);
	/**
	 * check if there are any weapons in storage
	 * @return
	 */
	boolean hasWeapons();
	/**
	 * Add or remove ware from the storage. If the amount is negative it will be removed.
	 * If the full amount is not available only the available will be moved and returned.
	 * The amount is in the ware specific size.
	 * The price of the wares is computed based on the amount
	 * @param ware
	 * @param amount
	 * @return amount moved
	 * @deprecated Use {@link #move(IWare, int, int)}
	 */
	@Deprecated
	int move(IWare ware, int amount);
	/**
	 * Add or remove weapon from the storage. If the amount is negative it will be removed.
	 * If the full amount is not available only the available will be moved and returned.
	 * The amount is in the ware specific size.
	 * @param weapons
	 * @param amount
	 * @return amount moved
	 */
	int move(IWeapon weapons, int amount);
	/**
	 * Retrieve the storage capacity of the trading office and all other storages in the same city
	 * @return storage capacity in barrels
	 */
	IntegerProperty capacityProperty();
	/**
	 * Move the ware into/from the storage with a price attached.
	 * @param ware to be moved
	 * @param moved amount to be moved
	 * @param avgPrice average price to use on the amount of ware
	 */
	int move(IWare ware, int moved, int avgPrice);
	/**
	 * Retrieve the balance sheet for last completed week.
	 * @return
	 */
	IBalanceSheet getBalanceLastWeek();
	/**
	 * Retrieve the balance sheet for the current week
	 * @return
	 */
	IBalanceSheet getCurrentWeek();
	/**
	 * Retrieve the steward of the trading office (manager). May be null if no manager is employed
	 * @return
	 */
	Optional<ISteward> getSteward();
	/**
	 * Setting the steward. May be null to indicate firing the steward
	 * @param steward
	 */
	void setSteward(ISteward steward);
	/**
	 * Get the trading object of the office
	 * @return
	 */
	IAutomatedTrading getOfficeTrading();
	/**
	 * Retrieve the weapon storage for this trading office
	 * @return
	 */
	IWeaponStorage getWeaponStorage();
	/**
	 * Retrieve an overview over the storage
	 * @return
	 */
	IStorage getStorage();
	/**
	 * Binding representing the current stored amount.
	 * @return
	 */
	IntegerBinding storedAmountBinding();

    /**
     * Property holding the steward.
     * @return
     */
    ObjectProperty<ISteward> stewardProperty();

    /**
     * Binding indicating the presence of a steward.
     * @return
     */
    BooleanBinding storageManagerPresentBinding();

	/**
	 * Retrieve tha amount of occupied space not available to the player.
	 * @return
	 */
	int getOccupiedSpace();

	/**
	 * Set the amount of occupied space not available to the player.
	 * @param space
	 */
	void setOccupiedSpace(int space);

	/**
	 * Retrieve the city the trading office is located in.
	 * @return
	 */
	ICity getCity();

	/**
	 * Replace the last weeks balance sheet with the current week and
	 * set the new balance sheet for this week.
	 * @param newWeek
     */
	void replaceBalanceSheet(IBalanceSheet newWeek);

	/**
	 * Retrieve the date when the trading office was established.
	 * @return
	 */
	LocalDateTime getEstablishedDate();
}
