package ch.sahits.game.openpatrician.model.building.impl;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.model.javafx.bindings.StaticIntegerBinding;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IBalanceSheet;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.IAutomatedTrading;
import ch.sahits.game.openpatrician.model.building.ISteward;
import ch.sahits.game.openpatrician.model.building.IStorage;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.IWeaponStorage;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.impl.WareHolding;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import ch.sahits.game.openpatrician.utilities.spring.DependentPropertyInitializer;
import ch.sahits.game.openpatrician.utilities.spring.DependentValue;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import lombok.Getter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.time.LocalDateTime;
import java.util.Optional;

/**
 * Implementation of the trading office
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Dec 17, 2011
 *
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class TradingOffice extends WareHolding implements ITradingOffice {
	@XStreamOmitField
	private static final Logger LOGGER = LogManager.getLogger(TradingOffice.class);
	/**
	 * Maximal capacity of the trading office in barrels
	 */
	@Value("${tradingoffice.max.capacity}")
	private int maxCapacity = 500;
	/**
	 * Reference to the player
	 */
    @Getter
	private final IPlayer owner;
	/** Reference to the city the trading office is located in */
	@Getter
	private final ICity city;
	/** Value of the building (building costs */
	private final int baseValue;
	/** Weapons storage */
    @Getter
	@Autowired
	private IWeaponStorage weaponStorage;
    @Getter
	@DependentValue("propertytax.tradingoffice")
    private int propertyTax = 5;
    @Getter
	private IStorage storage;
	@ObjectPropertyType(ISteward.class)
	private ObjectProperty<ISteward> steward;
    @Getter
	private IBalanceSheet balanceLastWeek;
    @Getter
	private IBalanceSheet currentWeek;
    @Getter
    private LocalDateTime establishedDate;
    @Getter
	@Autowired
	private IAutomatedTrading officeTrading;
	@Autowired
	@XStreamOmitField
	private DependentPropertyInitializer propertyInitializer;
	@Autowired
	private Date date;


	private IntegerProperty capacity = new SimpleIntegerProperty(this, "capacity", 0);
    @XStreamOmitField
	private BooleanBinding storageManagerPresent = null;
    // TODO aho Dec 7, 2013: consider capacity of warehouses

	public TradingOffice(IPlayer player, ICity city, int value, IStorage storage, IBalanceSheet lastWeek, IBalanceSheet thisWeek){
		this.owner=player;
		this.city = city;
		baseValue = value;
        steward = new SimpleObjectProperty<>(this, "steward", null);
		this.storage = storage;
		this.balanceLastWeek = lastWeek;
		this.currentWeek = thisWeek;
	}
	@PostConstruct
	private void init() {
		establishedDate = date.getCurrentDate();
		capacity.setValue(maxCapacity);
		try {
			propertyInitializer.initializeAnnotatedFields(this);
		} catch (IllegalAccessException e) {
			LOGGER.warn("Failed to initialize DependentValue annotated fields");
		}
		if (!containsWare(EWare.GRAIN)) {
			initWares();
		} else {
			bindWares();
		}
	}



	@PreDestroy
	public void destroy() {
		unbindAllAmounts();
	}
	/**
	 * Init the amount of wares available in the city
	 * This method is protected so it can be overriden by subclasses for testing
	 */
	private void initWares() {
		for (EWare ware : EWare.values()) {
				addNewWare(ware, 0);
		}

	}
	private void bindWares() {
		for (EWare ware : EWare.values()) {
			bindStoredAmount(getWare(ware).amountProperty());
		}
	}


	@Override
	public IntegerProperty capacityProperty() {
		return capacity;
	}



	@Override
	public boolean hasWeapons() {
		return weaponStorage.hasWeapons();
	}
	/**
	/**
	 * {@inheritDoc}
	 * This method is thread save.
	 */
	@Override
	public int move(IWeapon weapon, int amount) {
		synchronized (weapon) { // There is still a problem with networked solution where the model of the city is replicated.
			return weaponStorage.update(weapon, amount);
		}
	}
	/**
	 * Compute the average price for ware that is to be purchased in the specified
	 * amount. The price is computed based on the amount that is available in the city.
	 * @param ware for which the average price is to be computed
	 * @param amount amount that is bought
	 * @return
	 */
	@Override
	protected int computeAVGPrice(IWare ware, int amount) {
		int available = city.getWare(ware).getAmount();
		return computablePrice.sellPrice(ware, new SimpleIntegerProperty(available), new StaticIntegerBinding(amount));
	}
	/**
	 * {@inheritDoc}
	 */
	@Override
	public int move(IWare ware, int amount, int avgPrice) {
		synchronized (ware) { // There is still a problem with networked solution where the model of the city is replicated.
			AmountablePrice<IWare> amounted = getWare(ware);
			if (amount<0 && containsWare(ware) && amounted!=null && -amount>amounted.getAmount()){ // remove more than is available
				amount = -amounted.getAmount();
			}
			if (amount<0 && !containsWare(ware)){
				amount = 0;
			}
			if (!containsWare(ware)){ // If the ware does not exist add it with amount 0
				new IllegalStateException("All wares should be initialized, allowed in test").printStackTrace();
			}
			if (amount>0){ // amount was bought
				amounted.add(amount, avgPrice);
			} else { // amount sold
				amounted.remove(-amount);
			}
			return amount;
		}
	}

	@Override
	public Optional<ISteward> getSteward() {
		return Optional.ofNullable(steward.get());
	}
	@Override
	public void setSteward(ISteward steward) {
		this.steward.set(steward);
	}
    @Override
    public ObjectProperty<ISteward> stewardProperty() {
        return steward;
    }


	public void replaceBalanceSheet(IBalanceSheet newWeek) {
		balanceLastWeek = currentWeek;
		currentWeek = newWeek;
	}

	@Override
	public int move(IWare ware, int amount) {
		return super.move(ware, amount, owner);
	}

	@Override
	public int getValue() {
		return baseValue;
	}


    @Override
    public BooleanBinding storageManagerPresentBinding() {
        if (storageManagerPresent == null) {
            storageManagerPresent = new BooleanBinding() {
                {
                    super.bind(steward);
                }

                @Override
                protected boolean computeValue() {
                    return steward.get() != null;
                }
            };
        }
        return storageManagerPresent;
    }

	/**
	 * Owner of the trading office cannot be changed.
	 * @param newOwner
	 */
	@Override
	public void setOwner(IPlayer newOwner) {
		// NOP
	}
}
