package ch.sahits.game.openpatrician.model.city.cityhall.impl;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.OptionalType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.model.DateService;
import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.cityhall.AldermanCandidateList;
import ch.sahits.game.openpatrician.model.city.cityhall.IAldermanOffice;
import ch.sahits.game.openpatrician.model.city.cityhall.IBallot;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityGuard;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHall;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityHallNotice;
import ch.sahits.game.openpatrician.model.city.cityhall.ICityPetition;
import ch.sahits.game.openpatrician.model.city.cityhall.IElectionResult;
import ch.sahits.game.openpatrician.model.city.cityhall.IOutriggerContract;
import ch.sahits.game.openpatrician.model.city.cityhall.ITreasury;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.collections.ObservableList;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Dec 31, 2014
 */
@Prototype
@RequiredArgsConstructor
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class CityHall implements ICityHall {
    @Getter
    @ListType(ICityHallNotice.class)
    private final ObservableList<ICityHallNotice> notices;
    @Getter
    @Setter
    @OptionalType(IOutriggerContract.class)
    private Optional<IOutriggerContract> outriggerContract = Optional.empty();
    @Getter
    @Setter
    private ICitizen mayor;
    @Getter
    private LocalDateTime electionDate;
    @Getter
    @ListType(ICitizen.class)
    private final List<ICitizen> candidates;
    @Getter
    @OptionalType(LocalDateTime.class)
    private Optional<LocalDateTime> nextCouncilMeeting = Optional.empty();
    @Getter
    @Setter
    @OptionalType(ICityPetition.class)
    private Optional<ICityPetition> petition = Optional.empty();
    @Getter
    @Setter
    private ICitizen alderman;
    @Getter
    private LocalDateTime aldermanElectionDate;
    @Getter
    @Autowired
    private AldermanCandidateList aldermanCandidates;
    @Getter
    private final ITreasury treasury;
    @Getter
    @ListType(ICityGuard.class)
    private final List<ICityGuard> cityGuard;
    @Getter
    @Setter
    @OptionalType(IAldermanOffice.class)
    private Optional<IAldermanOffice> aldermanOffice = Optional.empty();
    @Getter
    private final ICity city;
    @Getter
    @ListType(ICitizen.class)
    private final List<ICitizen> councilmen;
    @Getter
    @OptionalType(LocalDateTime.class)
    private Optional<LocalDateTime> hanseaticMeetingDate = Optional.empty();
    @Getter
    @Setter
    @OptionalType(IElectionResult.class)
    private Optional<IElectionResult> electionResult = Optional.empty();
    @Getter
    @Setter
    @OptionalType(IBallot.class)
    private Optional<IBallot> ballotResult = Optional.empty();
    @Getter
    @Setter
    private int maxNumberMilita;
    @Autowired
    @XStreamOmitField
    private DateService date;

    /**
     * Set the elederman election date and verify that there are no clashed with election dates.
     * If there are the date is move as long as there are no clashes.
     * @param aldermanElectionDate
     */
    public void setAldermanElectionDate(LocalDateTime aldermanElectionDate) {
        LocalDateTime proposedDate = aldermanElectionDate;
        while (!isAldermanElectionDateOk(proposedDate)) {
            proposedDate = proposedDate.plusDays(1);
        }
        this.aldermanElectionDate = proposedDate;
    }

    /**
     * Set the election date and verify that there are no clashed with election dates.
     * If there are the date is move as long as there are no clashes.
     * @param electionDate
     */
    public void setElectionDate(LocalDateTime electionDate) {
        LocalDateTime proposedDate = electionDate;
        while (!isMayoralElectionDateOk(proposedDate)) {
            proposedDate = proposedDate.plusDays(1);
        }
        this.electionDate = proposedDate;
    }

    /**
     * Set the next council meeting date and verify that there are no clashed with election dates.
     * If there are the date is move as long as there are no clashes.
     * @param nextCouncilMeeting
     */
    public void setNextCouncilMeeting(Optional<LocalDateTime> nextCouncilMeeting) {
        if (nextCouncilMeeting.isPresent()) {
            LocalDateTime proposedDate = nextCouncilMeeting.get();
            while (!isCouncilMeetingDateOk(proposedDate)) {
                proposedDate = proposedDate.plusDays(1);
            }
            this.nextCouncilMeeting = Optional.of(proposedDate);
        } else {
            this.nextCouncilMeeting = nextCouncilMeeting;
        }
    }

    /**
     * Set the hanseatic meeting date and verify that there are no clashed with election dates.
     * If there are the date is move as long as there are no clashes.
     * @param meetingDate
     */
    public void setHanseaticMeetingDate(Optional<LocalDateTime> meetingDate) {
        if (meetingDate.isPresent()) {
            LocalDateTime proposedDate = meetingDate.get();
            while (!isHanseaticMeetingDateOk(proposedDate)) {
                proposedDate = proposedDate.plusDays(1);
            }
            this.hanseaticMeetingDate = Optional.of(proposedDate);
        } else {
            this.hanseaticMeetingDate = meetingDate;
        }
    }
    private boolean isAldermanElectionDateOk(LocalDateTime electionDate) {
        if (this.electionDate != null && date.isSameDay(electionDate, this.electionDate)) {
            return false;
        }
        if (nextCouncilMeeting.isPresent() && date.isSameDay(electionDate, nextCouncilMeeting.get())) {
            return false;
        }
        if (hanseaticMeetingDate.isPresent() && date.isSameDay(electionDate, hanseaticMeetingDate.get())) {
            return false;
        }
        return true;
    }
    private boolean isMayoralElectionDateOk(LocalDateTime electionDate) {
        if (this.aldermanElectionDate != null && date.isSameDay(electionDate, this.aldermanElectionDate)) {
            return false;
        }
        if (nextCouncilMeeting.isPresent() && date.isSameDay(electionDate, nextCouncilMeeting.get())) {
            return false;
        }
        if (hanseaticMeetingDate.isPresent() && date.isSameDay(electionDate, hanseaticMeetingDate.get())) {
            return false;
        }
        return true;
    }
    private boolean isCouncilMeetingDateOk(LocalDateTime meetingDate) {
        if (this.electionDate != null && date.isSameDay(meetingDate, electionDate)) {
            return false;
        }
        if (this.aldermanElectionDate != null && date.isSameDay(meetingDate, aldermanElectionDate)) {
            return false;
        }
        if (hanseaticMeetingDate.isPresent() && date.isSameDay(meetingDate, hanseaticMeetingDate.get())) {
            return false;
        }
        return true;
    }
    private boolean isHanseaticMeetingDateOk(LocalDateTime meetingDate) {
        if (this.electionDate != null && date.isSameDay(meetingDate, electionDate)) {
            return false;
        }
        if (this.aldermanElectionDate != null && date.isSameDay(meetingDate, aldermanElectionDate)) {
            return false;
        }
        if (nextCouncilMeeting.isPresent() && date.isSameDay(meetingDate, nextCouncilMeeting.get())) {
            return false;
        }
        return true;
    }
}
