package ch.sahits.game.openpatrician.model.ship.impl;

import ch.sahits.game.openpatrician.data.xmlmodel.BasicSlot;
import ch.sahits.game.openpatrician.data.xmlmodel.SlotType;
import ch.sahits.game.openpatrician.data.xmlmodel.WeaponLocation;
import ch.sahits.game.openpatrician.data.xmlmodel.WeaponLocations;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.ESide;
import ch.sahits.game.openpatrician.model.ship.IWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.PrimaryLargeWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.SecondaryLargeWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.SmallWeaponSlot;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.oxm.Unmarshaller;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Ship weapons location factory.
 * Weapons locations are a list of weapon slots. They are either intended for 'small' or 'big' weapons.
 * A big weapons take up a primary and a secondary slot, while small weapons can be placed in either.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jun 1, 2013
 *
 */
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class ShipWeaponsLocationFactory {
	private final Logger logger = LogManager.getLogger(getClass());

	@Autowired
	@Qualifier("jaxb2XmlModelMarshaller")
	private Unmarshaller unmarshaller;

	private WeaponLocations weaponLocations;

	@PostConstruct
	private void initilizeWeaponModel() {
		try {
			Source sourceFromFile = getSourceFromFile("/weaponLocation.xml");
			weaponLocations = (WeaponLocations) unmarshaller.unmarshal(sourceFromFile);
		} catch (IOException e) {
			logger.warn("Failed to initialize weapons location");
		}
	}

	private Source getSourceFromFile(String fileName) {
		final InputStream resourceAsStream = getClass().getResourceAsStream(fileName);
		return new StreamSource(resourceAsStream);
	}

	public List<IWeaponSlot> getShipWeaponsLocation(EShipType type, EShipUpgrade level) {
		WeaponLocation weaponLocation = null;
		for (WeaponLocation location : weaponLocations.getWeaponLocation()) {
			if (type.name().equals(location.getType().name()) && checkLevel(level, location)) {
				weaponLocation = location;
				break;
			}
		}
		return convert(weaponLocation);
	}

	private boolean checkLevel(EShipUpgrade level, WeaponLocation location) {
		switch (location.getLevel()) {
			case NONE:
				return level.name().equals(location.getLevel().name());
			case LEVEL_1:
				return level.name().equals("LEVEL1");
			case LEVEL_2:
				return level.name().equals("LEVEL2");
		}
		return false;
	}

	private List<IWeaponSlot> convert(WeaponLocation weaponLocation) {
		List<IWeaponSlot> weaponSlots = new ArrayList<>();
		for (Iterator<BasicSlot> iterator = weaponLocation.getSlot().iterator(); iterator.hasNext(); ) {
			BasicSlot next = iterator.next();
			IWeaponSlot slot = convert(next);
			weaponSlots.add(slot);
		}
		return weaponSlots;
	}

	private IWeaponSlot convert(BasicSlot slot) {
		if (slot.getType().equals(SlotType.SMALL)) {
			SmallWeaponSlot small = new SmallWeaponSlot(ESide.valueOf(slot.getSide().name()), slot.getId());
			small.setOffset(new Point2D( slot.getX(), slot.getY()));
			return small;
		}  else if (slot.getType().equals(SlotType.PRIMARY)) {
			PrimaryLargeWeaponSlot primary = new PrimaryLargeWeaponSlot(ESide.valueOf(slot.getSide().name()), slot.getId());
			primary.setOffset(new Point2D( slot.getX(), slot.getY()));
			return primary;
		} else if (slot.getType().equals(SlotType.SECONDARY)) {
			SecondaryLargeWeaponSlot secondary = new SecondaryLargeWeaponSlot(ESide.valueOf(slot.getSide().name()), slot.getId());
			secondary.setOffset(new Point2D( slot.getX(), slot.getY()));
			return secondary;
		} else {
			throw new IllegalArgumentException("Unrecognized type "+slot.getClass());
		}
	}

}
