package ch.sahits.game.openpatrician.model;

import ch.sahits.game.event.data.ClockTickIntervalChange;
import ch.sahits.game.openpatrician.utilities.IInvalidatable;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import lombok.Getter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;

/**
 * Representation of the date within the game.
 * This class is thread safe.
 * The Date is implemented as a singleton. However there
 * is no getInstance method to retrieve the once created instance,
 * so it must be referenced elsewhere.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 16, 2011
 *
 */
@Component
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN,EClassCategory.SINGLETON_BEAN})
public class Date implements IInvalidatable {
	@XStreamOmitField
	private static final Logger LOGGER = LogManager.getLogger(Date.class);

	public static final int DEFAULT_INVALID_START_YEAR = 0;
	/** Tick update in minutes */
	@Getter
    private int tickUpdate;
	/** Current date */
	@ObjectPropertyType(LocalDateTime.class)
	private ObjectProperty<LocalDateTime> cal = new SimpleObjectProperty<>(this, "cal", null);;
	@Getter
    private LocalDateTime startDate;
	/** Start year */
    @Getter
	private Integer startYear = null;
	/** Lock for guaranteeing thread safety */
	private static Object lock = new Object();
	@XStreamOmitField
	private StringBinding dayDateBinding = null;
	@Autowired
	@Qualifier("serverClientEventBus")
	@XStreamOmitField
	private AsyncEventBus clientServerEventBus;
	@Autowired
	@XStreamOmitField
	private IModelTranslationService modelTranslationService;
	@Autowired
	@XStreamOmitField
	private DateService dateService;

	/**
	 * Initialize the date with start year.
	 * The initial date is 13th July of the start year.
	 * The default tick update is 5 minutes.
	 */
	public Date(int year){
		this();
		setStartYear(year);
	}


	public Date() {
		tickUpdate = 5;
	}
	public void resetStartYear(int startYear) {
		LOGGER.debug("Set the start year: {}", startYear);
		this.startYear=startYear;
		updateTime(LocalDateTime.of(startYear, 7, 13, 0, 0));
		startDate = getCurrentDate();
	}
	public void setStartYear(int startYear) {
		Preconditions.checkState(checkStartYear(startYear), "Date may only be initialized once");
		resetStartYear(startYear);
	}
	private boolean checkStartYear(int startYear) {
		if (this.startYear == null) {
			return true;
		}
		if (this.startYear == DEFAULT_INVALID_START_YEAR) {
			return true;
		}
		return false;
	}
	/**
	 * Constructor for testing purposes only
	 * @param cal initial date
	 */
	@VisibleForTesting
	Date(LocalDateTime cal){
		updateTime(cal);
		startYear = cal.getYear();
		tickUpdate = 5;
	}

    /**
     * Change the time.
     * @param newTime new time
     */
	public void updateTime(LocalDateTime newTime) {
		if (cal.get() != null && !dateService.isToday(newTime) && dayDateBinding != null) {
			dayDateBinding.invalidate();
		}
		Preconditions.checkNotNull(newTime);
		cal.set(newTime);
    }

	/**
	 * Set the tick update in number of minutes
	 * @param minutes tick update
	 */
	public void setTickUpdate(int minutes){
		tickUpdate =minutes;
		final ClockTickIntervalChange event = new ClockTickIntervalChange();
		event.setInterval(tickUpdate);
		clientServerEventBus.post(event);
	}

    /**
     * Bind the representation of the day as a String.
     * @return string binding for the current date
	 * @see ch.sahits.game.openpatrician.model.service.ModelTranslations#toDisplayString(LocalDateTime)
     */
    public StringBinding dayDateBinding() {
          if (dayDateBinding == null) {
              dayDateBinding = new StringBinding() {
                  @Override
                  protected String computeValue() {
                      return modelTranslationService.toDisplayString(cal.get());
                  }
              };
          }
        return dayDateBinding;
    }

	/**
	 * Retrieve the immutable DateTime
	 * @return current date
	 */
	public LocalDateTime getCurrentDate(){
		return cal.get();
	}
	/**
	 * Retrieve the immutable DateTime
	 * @return observable of the current date,
	 */
	public ObjectProperty<LocalDateTime> getCurrentDateProperty(){
		return cal;
	}


    @Override
    public void invalidate() {
        dayDateBinding.invalidate();
        dayDateBinding = null;
        dayDateBinding();
    }
}
