package ch.sahits.game.openpatrician.model;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import com.google.common.base.Preconditions;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.LocalDateTime;
import java.time.Period;
import java.time.temporal.ChronoUnit;

/**
 * Service concerning the date.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on May 15, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class DateService {

    @Autowired
    private Date date;

    /**
     * Check if two dates represent the same day
     * @param date1 first date to compair
     * @param date2 second date to compair
     * @return true if both dates are on the same day
     */
    public boolean isSameDay(LocalDateTime date1, LocalDateTime date2) {
        return date1.getDayOfYear() == date2.getDayOfYear() && date1.getYear() == date2.getYear();
    }
    /**
     * Check if the date is today.
     * @param date to chack against the current date
     * @return true if <code>date</code> is on the same day as current date
     */
    public boolean isToday(LocalDateTime date) {
        return isSameDay(date, this.date.getCurrentDate());
    }

    /**
     * Check if a date lies in the past.
     * @param date to be checked.
     * @return true if the date is before the current game date
     */
    public boolean isPast(LocalDateTime date) {
        return date.isBefore(this.date.getCurrentDate());
    }

    /**
     * Calculate the age.
     * @param birthDate from which to calculate the age
     * @return age in years
     */
    public int getAge(LocalDateTime birthDate) {
        return Period.between(birthDate.toLocalDate(), date.getCurrentDate().toLocalDate()).getYears();
    }

    /**
     * Calculate the percentage of days passed between the start and end date.
     * @param start date
     * @param end date
     * @return percentage of days passed in [0,1]
     */
    public double getPercentageBetweenDates(LocalDateTime start, LocalDateTime end) {
        Preconditions.checkArgument(end.isAfter(start), "End date must be after start date");
        double duration =  ChronoUnit.DAYS.between(start, end);
        double pastDays =  ChronoUnit.DAYS.between(start, date.getCurrentDate());
        return pastDays/duration;
    }

}
