package ch.sahits.game.openpatrician.model.city.cityhall;

import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.city.ICity;
import javafx.collections.ObservableList;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

/**
 * Representing the state of the various issues that may be addressed
 * at the city hall.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Dec 31, 2014
 */
public interface ICityHall {
    /**
     * Retrieve the list of notices for wares that can be delivered
     * to other cities at a guaranteed price. These notices get updated
     * on a weekly basis.
     * @return observable list of all notices in the city hall
     */
    ObservableList<ICityHallNotice> getNotices();

    /**
     * Current outrigger of the city. As the city may not have an outrigger
     * assigned it may be absent.
     * @return Optional of the outrigger contract. empty if no outrigger is assigned
     */
    Optional<IOutriggerContract> getOutriggerContract();

    /**
     * Set the outrigger contract.
     * @param contract set the Optional outrigger contract
     */
    void setOutriggerContract(Optional<IOutriggerContract> contract);

    /**
     * Current mayor of the city
     * @return mayor of the city.
     */
    ICitizen getMayor();

    /**
     * Date of the next election. Elections happen all 2 years.
     * @return date of the next election
     */
    LocalDateTime getElectionDate();

    /**
     * Retrieve the list of candidates for the next election.
     * @return List of mayor candidates
     */
    List<ICitizen> getCandidates();

    /**
     * Date of the next council meeting to discuss a petition.
     * @return Optional date of the next council meeting. Empty if there is no council meeting.
     */
    Optional<LocalDateTime> getNextCouncilMeeting();

    /**
     * Petition to be discussed at the next council meeting.
     * @return Optional petition. Empty if there is no petition.
     */
    Optional<ICityPetition> getPetition();

    /**
     * Set the petition.
     * @param petition Optional petition to set.
     */
    void setPetition(Optional<ICityPetition> petition);

    /**
     * Retrieve the alderman.
     * @return alderman of the Hanseatic League
     */
    ICitizen getAlderman();

    /**
     * Date of the next alderman election.
     * @return date of the next alderman election
     */
    LocalDateTime getAldermanElectionDate();

    /**
     * Retrieve the top four of the candidates for the alderman post.
     * @return list of alderman candidates
     */
    AldermanCandidateList getAldermanCandidates();

    /**
     * Retrieve the treasury.
     * @return treasury of the city
     */
    ITreasury getTreasury();

    /**
     * Retrieve the city guard.
     * @return list of city guards
     */
    List<ICityGuard> getCityGuard();

    /**
     * Retrieve the office of the alderman. This office is
     * only available if the alderman resides in the currenct city.
     * @return Optional alderman office. Empty if the office is located in another city.
     */
    Optional<IAldermanOffice> getAldermanOffice();

    /**
     * Retrieve the city belonging to this city hall
     * @return corresponding city.
     */
    ICity getCity();


    /**
     * Retrieve the list of all councilmen of the city
     * @return List of citizens on the council.
     */
    List<ICitizen> getCouncilmen();
    /**
     * Retrieve the meeting date for the next extra circular hanseatic day.
     * @return Optional date of the next Hanseatic Meeting. Empty if there is no meeting
     */
    Optional<LocalDateTime> getHanseaticMeetingDate();

    /**
     * Retrieve the Optional election result.
     * @return Optional election result, Empty if there is no election
     */
    Optional<IElectionResult> getElectionResult();

    /**
     * Retrieve the ballot result
     * @return Optional ballot result. Empty if there is no ballot
     */
    Optional<IBallot> getBallotResult();

    /**
     * Retrieve the maximum allowed members of the guard.
     * @return max number of city guards.
     */
    int getMaxNumberMilita();

}
