package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.datastructure.GenericPair;
import ch.sahits.game.event.data.ClockTickPostDayChange;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.city.IBuildingProduction;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.city.PopulationConsume;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.spring.DependentPropertyInitializer;
import ch.sahits.game.openpatrician.utilities.spring.DependentValue;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.ObjectBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import lombok.Getter;
import lombok.Setter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.TreeMap;

/**
 * This class encapsulates state information used by CityEngine about one city.
 * This class i player private since it should not be used by any other class than CityEngine.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 29, 2011
 *
 */
@Component
@Scope("prototype")
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class CityState {
	@XStreamOmitField
    private static final Logger LOGGER = LogManager.getLogger(CityState.class);

	@Autowired
	@XStreamOmitField
	private Random rnd;
	@SuppressWarnings("unused")
	@DependentValue("ware.surplus.treshold")
	private double surplusThreshold;

	/** Reference to the city model that is driven by this engine */
    @Getter
	private ICity city;
	/** Map of consumed wares, since only whole units can be consumed we need to store the fractions */
	@MapType(key=IWare.class, value=Double.class)
	private Map<IWare, Double> consumed = new HashMap<>();
	/** Tavern state for this city */
    @Getter
	private TavernState tavernState;
	/** Ship yard state fos the city */
    @Getter
	private IShipyard shipyardState;
	/** Provider for the consume numbers */
	@Autowired
	private PopulationConsume consume;
	@Autowired
	@XStreamOmitField
	private IBuildingProduction buildingProduction;

	@Autowired
	@XStreamOmitField
	private DependentPropertyInitializer propertyInitializer;
	@Getter
    @Setter
    @Autowired
    private CityWall cityWall;
	@Autowired
	@Getter
	private PopulationUpdateStats popUpdateStatistic;
	@Autowired
	@Qualifier("timerEventBus")
	@XStreamOmitField
	private AsyncEventBus timerEventBus;
	@Autowired
	private Date date;
	@MapType(key = IWare.class, value = LocalDateTime.class)
	private Map<IWare, LocalDateTime> longestMissingWares = new HashMap<>();
	@ObjectPropertyType(IWare.class)
	private ObjectBinding<IWare> longestMissingWare = null;

	private BooleanBinding famine;
	@ObjectPropertyType(ECityState.class)
	private ObjectProperty<ECityState> cityState = new SimpleObjectProperty<>(null);

	public CityState(ICity city, IShipyard shipard, TavernState tavern) {
		this.city = city;
		this.shipyardState = shipard;
		this.tavernState = tavern;
	}
	@PostConstruct
	private void initCity() {
		EWare[] wares = EWare.values();
		for (EWare ware : wares) {
			replaceConsumedAmount(ware, 0.0);
		}
		if (city.getCityState() == null) {
			city.setCityState(this);
		}
		try {
			propertyInitializer.initializeAnnotatedFields(this);
		} catch (IllegalAccessException e) {
			LOGGER.warn("Failed to initialize DependentValue annotated fields");
		}
		longestMissingWaresBinding();
		dailyUpdate(null);
		timerEventBus.register(this);
	}

	/**
	 * Update the missing wares.
	 * @param dayChange event indicating a day change
	 */
	@Subscribe
	public void dailyUpdate(ClockTickPostDayChange dayChange) {
		List<IWare> wares = new ArrayList<>(Arrays.asList(EWare.values()));
		Collections.shuffle(wares);
		LocalDateTime now = date.getCurrentDate();
		boolean changed = false;
		for (IWare ware : wares) {
			if (city.getWare(ware).getAmount() <= 0) {
				if (!longestMissingWares.containsKey(ware)) {
					longestMissingWares.put(ware, now);
					changed = true;
				}
			} else {
				// ware is not missing
				if (longestMissingWares.remove(ware) != null) {
					changed = true;
				}
			}
		}
		if (changed && longestMissingWare != null) {
			longestMissingWare.invalidate();
		}
	}

	/**
	 * Retrieve the ware that is missing the longest.
	 * @return ware that is missing the longest
	 */
	private IWare getLongestMissingWares() {
		GenericPair<IWare, LocalDateTime> oldest = new GenericPair<>(null, date.getCurrentDate());
		for (Entry<IWare, LocalDateTime> entry : longestMissingWares.entrySet()) {
			if (entry.getValue().isBefore(oldest.getSecond())) {
				oldest = new GenericPair<>(entry.getKey(), entry.getValue());
			}
		}
		return oldest.getFirst();
	}

	/**
	 * Retrieve the ware that is missing the longest.
	 * @return ware binding for the ware missing longest
	 */
	public ObjectBinding<IWare> longestMissingWaresBinding() {
		if (longestMissingWare == null) {
			longestMissingWare = new ObjectBinding<>() {
				@Override
				protected IWare computeValue() {
					return getLongestMissingWares();
				}
			};
		}
		return longestMissingWare;
	}

    /**
     * Check if there is a famine. Famine happens if there is no grain for two weeks.
     * @return true if there is a famine
     */
	private boolean isFamine() {
	    LocalDateTime twoWeeksAgo = date.getCurrentDate().minusWeeks(2);
	    if (longestMissingWares.containsKey(EWare.GRAIN)) {
	        return longestMissingWares.get(EWare.GRAIN).isBefore(twoWeeksAgo);
        }
        return false;
    }

	/**
	 * Binding indicating if there is a famine.
	 * @return boolean binding for the famine
	 */
	public BooleanBinding famineBinding() {
		if (famine == null) {
			famine = new BooleanBinding() {
				{
					super.bind(longestMissingWaresBinding());
				}
				@Override
				protected boolean computeValue() {
					return isFamine();
				}
			};
		}
		return famine;
	}

	/**
	 * Property holding the corrent city event, may be null.
	 * @return binding of the city event state
	 */
	public ObjectProperty<ECityState> cityEventProperty() {
		return cityState;
	}
	/**
	 * Get the consumed entries as a set.
	 * @return set of entries of wares and their consumption
     */
	public Set<Entry<IWare, Double>> consumedEntries() {
		return consumed.entrySet();
	}

	/**
	 * Set the amount of the ware that is consumed. If there is already an amount
	 * stored for the ware it is overidden.
	 * @param ware to be consumed
	 * @param amount to be consumed
     */
	public void replaceConsumedAmount(IWare ware, double amount) {
		consumed.put(ware, amount);// does this produce ConcurrentModExc?
	}

	/**
	 * Retrieve the amount that is consumed so far.
	 * @param ware which should be checked for consumption.
	 * @return consumed amount.
     */
	public double getConsumedAmount(EWare ware) {
		return consumed.get(ware);
	}


	/**
	 * Find the ware which is consumed the most and which is missing for the longest time.
	 * @return Optional ware
	 */
	public Optional<IWare> findMostNeededWare() {
		Map<IWare, LocalDateTime> missing = city.getMissingWares();
		LocalDateTime earliest = null;
		IWare result = null;
		for (Entry<IWare, LocalDateTime> entry : missing.entrySet()) {
			if (earliest == null ||
					entry.getValue().isBefore(earliest)) {
				earliest = entry.getValue();
				result = entry.getKey();
			}
		}
        if (result == null) { // All wares are availalbe => Look which one will run out first
            Map<IWare, Integer> available = getSortedWareAvailabilityMap();
            IWare wareRunninOutFirst = null;
            int runsOutInDays = Integer.MAX_VALUE;
            for (Entry<IWare, Integer> entry : available.entrySet()) {
                IWare ware = entry.getKey();
                double consumptionPerWeek = consume.getWeeklyConsumption(ware, city);
                double productionPerWeek = buildingProduction.getTotalProduction(ware, getCity());
                double dailyConsumtion = (productionPerWeek - consumptionPerWeek)/7;
                if (dailyConsumtion >= 0) {
                    continue; // more ware is produced than consumed
                }
                int days = -(int)(entry.getValue()/dailyConsumtion);
                if (days < runsOutInDays) {
                    runsOutInDays = days;
                    wareRunninOutFirst = ware;
                }
            }
            result = wareRunninOutFirst;
        }

		return Optional.ofNullable(result);
	}
	/**
	 * Find the ware that the city sells near the minimal price.
	 * @return Optional ware
	 */
	public Optional<IWare> findWareWithMostSurplus() {
		ArrayList<IWare> surplus = new ArrayList<>();
		for (IWare ware : EWare.values()) {
			AmountablePrice<IWare> amount = city.getWare(ware);
			double actualPrice = amount.getAVGPrice();
			int minPrice = ware.getMinValueSell();
			double val = minPrice/actualPrice - 1;
			if (val<=surplusThreshold) {
				surplus.add(ware);
			}
		}
		if (surplus.isEmpty()) {
			return Optional.empty();
		} else {
			return Optional.of(surplus.get(rnd.nextInt(surplus.size())));
		}
	}

    /**
     * Find the wares and order them in the amount of their availability. If there
     * is a ware with the same availability, only one is added to the map.
     * @return map of wares and their amounts available in the city.
     */
    @VisibleForTesting
    Map<IWare, Integer> getSortedWareAvailabilityMap() {
        Comparator<IWare> comparator = (ware1, ware2) -> city.getWare(ware1).getAmount() - city.getWare(ware2).getAmount();
        TreeMap<IWare, Integer> map = new TreeMap<>(comparator);
        for (IWare ware : EWare.values()) {
            map.put(ware, city.getWare(ware).getAmount());
        }
        map.size(); // this seems to be required to ensure the proper size is retrieved.
        return map;
    }
}
