package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipDueDate;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapOptionalValueType;
import ch.sahits.game.openpatrician.utilities.spring.DependentPropertyInitializer;
import ch.sahits.game.openpatrician.utilities.spring.DependentValue;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Getter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Random;

import static com.google.common.collect.Maps.newHashMap;
/**
 * Implementation of the shipyard model.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Mar 9, 2013
 *
 */
@Component
@Scope("prototype")
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class ShipyardState implements IShipyard {
    @XStreamOmitField
	private static final Logger LOGGER = LogManager.getLogger(ShipyardState.class);
	@DependentValue("shipyard.build.time.snaikka")
	private int buildTimeSnaikka;
	@DependentValue("shipyard.build.time.crayer")
	private int buildTimeCrayer;
	@DependentValue("shipyard.build.time.cog")
	private int buildTimeCog;
	@DependentValue("shipyard.build.time.holk")
	private int buildTimeHolk;
	@DependentValue("shipyard.repair.time.snaikka")
	private int repairTimeSnaikka;
	@DependentValue("shipyard.repair.time.crayer")
	private int repairTimeCrayer;
	@DependentValue("shipyard.repair.time.cog")
	private int repairTimeCog;
	@DependentValue("shipyard.repair.time.holk")
	private int repairTimeHolk;
	@Value("${shipyard.material.cost.per.unit}")
	private int materialUnitCost;
	@Value("${shipyard.salary.per.day}")
	private int baseSalaryPerDay;
	@MapOptionalValueType(key=IShipDueDate.class, value = TimedTask.class)
	private final Map<IShipDueDate, Optional<TimedTask>> repairList = newHashMap();
	@MapOptionalValueType(key=IShipDueDate.class, value = TimedTask.class)
	private final Map<IShipDueDate, Optional<TimedTask>> buildList = newHashMap();
	@MapOptionalValueType(key=IShipDueDate.class, value = TimedTask.class)
	private final Map<IShipDueDate, Optional<TimedTask>> refitList = newHashMap();

	/** Value between 0 and 1 detailing the experience of the ship yard crew (the higher the better) */
	private double experience;
	@Autowired
	private Date date;
	@Autowired
	@XStreamOmitField
	private Random rnd;
    @Getter
	private final ICity city;
	@Autowired
	private TimedUpdatableTaskList taskList;
	@Autowired
	private IMap map;
	@Autowired
	@XStreamOmitField
	private DependentPropertyInitializer propertyInitializer;

	public ShipyardState(ICity city) {
		this.city = city;
	}

	@PostConstruct
	public void initializeExperience() {
		try {
			propertyInitializer.initializeAnnotatedFields(this);
		} catch (IllegalAccessException e) {
			LOGGER.warn("Failed to initialize DependentValue annotated fields");
		}
		int mapWidth = (int) map.getDimension().getWidth();
		double eastFactor = city.getCoordinates().getX()/mapWidth/0.5;
		double startYearPart = Math.max((date.getStartYear()-1430)/100,0);
		experience = Math.max(startYearPart-eastFactor,0);
		Preconditions.checkArgument(experience>=0, "Experiance must be positive");
	}

	@Override
	public List<IShipDueDate> getShipBuildingList() {
		return ImmutableList.copyOf(buildList.keySet());
	}

	@Override
	public List<IShipDueDate> getShipUpgradeList() {
		return ImmutableList.copyOf(refitList.keySet());
	}


    @Override
    public void addShipBuildingOrder(IShipDueDate dueDate, Optional<TimedTask> task) {
        buildList.put(dueDate, task);
    }

	@Override
	public void cancelShipBuildingOrder(LocalDateTime finish) {
		for (Iterator<IShipDueDate> iterator = buildList.keySet().iterator(); iterator.hasNext();) {
			IShipDueDate dueDate = iterator.next();
			if (dueDate.getDueDate().equals(finish)) {
				Optional<TimedTask> task = buildList.get(dueDate);
				task.ifPresent(timedTask -> taskList.remove(timedTask));
				iterator.remove();
				break;
			}
		}
	}

	@Override
	public List<IShipDueDate> getShipRepairList() {
		return ImmutableList.copyOf(repairList.keySet());
	}


    @Override
    public void addShipRepairOrder(IShipDueDate dueDate, Optional<TimedTask> task) {
        repairList.put(dueDate, task);
        experience = Math.max(experience+0.005, 1);
    }

    @Override
	public int calculateRepairMaterialCosts(IShip ship, int damage) {
		double shipSize = getShipSize(ship);
		return (int)(damage*10*shipSize* materialUnitCost);
	}

	private double getShipSize(IShip ship) {
		double shipSize = 0;
		switch (ship.getShipType()) {
		case SNAIKKA:
			shipSize = 1;
			break;
		case CRAYER:
			shipSize = 1.5;
			break;
		case COG:
			shipSize = 2.5;
			break;
		case HOLK:
			shipSize = 5;
			break;
		}
		return shipSize;
	}
	@Override
	public int calculateRepairCosts(int repairTime, int materialCosts) {
		return repairTime*calculateSaleryPerDay()+materialCosts;
	}

	/**
	 * Salery is defined by the experiance
	 * @return salery per day.
	 */
	private int calculateSaleryPerDay() {
		return (int) (baseSalaryPerDay *(1+experience*10));
	}
	@Override
	public void cancelRepair(IShip ship) {
		for (Iterator<IShipDueDate> iterator = repairList.keySet().iterator(); iterator.hasNext();) {
			IShipDueDate dueDate = iterator.next();
			if (dueDate.getShip().equals(ship)) {
				Optional<TimedTask> task = repairList.get(dueDate);
				task.ifPresent(timedTask -> taskList.remove(timedTask));
				iterator.remove();
				ship.setAvailable(true);
				break;
			}
		}
	}


    @Override
    public void addShipRefitOrder(IShipDueDate dueDate, Optional<TimedTask> task) {
        refitList.put(dueDate, task);
    }

    @Override
	public int calculateRefitTime(EShipType type) {
		return calculateBuildTime(type)/10;
	}

	@Override
	public LocalDateTime getBuildCompleteDate(EShipType type) {
		LocalDateTime lastOrderFinished = date.getCurrentDate();
		for (IShipDueDate dueDate : buildList.keySet()) {
			if (dueDate.getDueDate().isAfter(lastOrderFinished)) {
				lastOrderFinished = dueDate.getDueDate();
			}
		}
		int buildTime = calculateBuildTime(type);
		return lastOrderFinished.plusDays(buildTime);
	}
	@Override
	public int calculateBuildTime(EShipType type) {
		int baseBuildTime = getBaseRepairTime(type);
		return (int) Math.rint(baseBuildTime+(1-experience)*baseBuildTime);
	}
	@Override
	public int calculateRepairTime(EShipType type) {
		int baseRepairTime = getBaseRepairTime(type);
		return (int) Math.rint(baseRepairTime+(1-experience)*baseRepairTime);
	}

	private int getBaseRepairTime(EShipType type) {
		switch (type) {
		case SNAIKKA:
			return repairTimeSnaikka;
		case CRAYER:
			return repairTimeCrayer;
		case COG:
			return repairTimeCog;
		case HOLK:
			return repairTimeHolk;
		}
		return 0;
	}

//	/**
//	 * Remove entries that lie in the past. This are orders that are fullfiled and
//	 * can be removed.
//	 */
//	public void cleanup() {
//		DateTime now = date.getCurrentDate();
//		for (Iterator<IShipDueDate> iterator = buildList.keySet().iterator(); iterator.hasNext();) {
//			IShipDueDate dueDate = iterator.next();
//			if (dueDate.getDueDate().isBefore(now)) {
//				LOGGER.debug("Remove {} with dueDate {} from buildList", dueDate.getShip().getName(), dueDate.getDueDate());
//				iterator.remove();
//			}
//		}
//		for (Iterator<IShipDueDate> iterator = repairList.keySet().iterator(); iterator.hasNext();) {
//			IShipDueDate dueDate = iterator.next();
//			if (dueDate.getDueDate().isBefore(now)) {
//				LOGGER.debug("Remove {} with dueDate {} from repairList", dueDate.getShip().getName(), dueDate.getDueDate());
//				iterator.remove();
//			}
//		}
//		for (Iterator<IShipDueDate> iterator = refitList.keySet().iterator(); iterator.hasNext();) {
//			IShipDueDate dueDate = iterator.next();
//			if (dueDate.getDueDate().isBefore(now)) {
//				LOGGER.debug("Remove {} with dueDate {} from refitList", dueDate.getShip().getName(), dueDate.getDueDate());
//				iterator.remove();
//			}
//		}
//	}


	@Override
	public int calculateRefitCosts(EShipType type, int levels) {
		return calculateRefitTime(type)*levels*calculateSaleryPerDay();
	}
	@Override
	public int calculateConstructionCosts(EShipType type) {
		return calculateBuildTime(type)*calculateSaleryPerDay();
	}

	@Override
	public EShipType[] getBuildableShipTypes() {
		// Cog 1420 + experiance > 0.2
		// Holk 1450 + expertice > 0.4
		if (date.getCurrentDate().getYear() > 1450 && experience > 0.4) {
			return new EShipType[]{EShipType.SNAIKKA, EShipType.CRAYER, EShipType.COG, EShipType.HOLK};
		} else if (date.getCurrentDate().getYear() > 1420 && experience > 0.2) {
			return new EShipType[]{EShipType.SNAIKKA, EShipType.CRAYER, EShipType.COG};
		} else {
			return new EShipType[]{EShipType.SNAIKKA, EShipType.CRAYER};
		}
	}

	@Override
	public void removeCompletedRepair(IShip ship) {
		removeEntry(ship, refitList);
	}

	private void removeEntry(IShip ship, Map<IShipDueDate, Optional<TimedTask>> list) {
		for (Iterator<Entry<IShipDueDate, Optional<TimedTask>>> iterator = list.entrySet().iterator(); iterator.hasNext(); ) {
			Entry<IShipDueDate, Optional<TimedTask>> next = iterator.next();
			if (next.getKey().getShip().equals(ship)) {
				iterator.remove();
				break;
			}
		}
	}

	@Override
	public void removeCompletedConstruction(IShip ship) {
		removeEntry(ship, buildList);
	}

	@Override
	public void removeCompletedUpgrade(IShip ship) {
		removeEntry(ship, refitList);

	}
}
