package ch.sahits.game.openpatrician.model.impl;

import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.map.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.personal.EEconomicCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import ch.sahits.game.openpatrician.model.personal.IChild;
import ch.sahits.game.openpatrician.model.personal.IPersonalData;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.IgnoreOnDeserialisation;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.service.LUIDProvider;
import ch.sahits.util.ClassChecker;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import static com.google.common.collect.Lists.newArrayList;
public abstract class Player implements IPlayer {
	@ListType(IShip.class)
	private final ObservableList<IShip> fleet;
	@ListType(INavigableVessel.class)
	private final ObservableList<INavigableVessel> selectableVessel;
	@Getter
	private String uuid;
    @Getter
	private ESocialRank rank;
    @Getter
    @Setter
	private ICareer careerLevel;
	private IPersonalData personalData;

	private Optional<ISpouseData> spouse = Optional.empty();
    @Getter
	private final ICompany company;
	@Getter
	@Setter
	private MapSegmentedImage segmentedMap;
	@Autowired
	@Qualifier("serverClientEventBus")
	@XStreamOmitField
	private AsyncEventBus clientServerEventBus;
	@Autowired
	@XStreamOmitField
	private LUIDProvider luidProvider;
	@Getter
	private int criminalDrive = 0;
	@ListType(IChild.class)
	@Getter
	private List<IChild> children;



	public Player(ICity homeTown, IPersonalData personalData,ESocialRank rank, ICompany company) {
		super();
		fleet = FXCollections.observableArrayList();
		selectableVessel = FXCollections.observableArrayList();
		this.rank=rank;
		careerLevel=EEconomicCareer.INEXPERIENCED;
		this.personalData=personalData;
		this.company = company;
		children = new ArrayList<>();

	}
	@PostConstruct
	@IgnoreOnDeserialisation
	private void init() {
		uuid = luidProvider.getNextLUID();
	}

	@Override
	public String getName() {
		return personalData.getName();
	}

	@Override
	public String getLastName() {
		return personalData.getLastName();
	}

	@Override
	public ICity getHometown() {
		return company.getHomeTown();
	}

	@Override
	public List<IShip> getFleet() {
		return Collections.unmodifiableList(fleet);
	}

	@Override
	public List<INavigableVessel> getSelectableVessels() {
		return Collections.unmodifiableList(selectableVessel);
	}

	@Override
	public void addSelectableVessel(INavigableVessel vessel) {
		Preconditions.checkArgument(this.equals(vessel.getOwner()), "The ships owner is not this player");
	   selectableVessel.add(vessel);
	}

	@Override
	public void removeSelectableVessel(INavigableVessel vessel) {
		Preconditions.checkArgument(this.equals(vessel.getOwner()), "The ships owner is not this player");
		selectableVessel.remove(vessel);
	}

	@Override
	public void addShip(IShip ship) {
		Preconditions.checkArgument(this.equals(ship.getOwner()), "The ships owner is not this player");
		fleet.add(ship);
	}

	@Override
	public void removeShip(IShip ship) {
		fleet.remove(ship);
	}

	@Override
	public List<IBuilding> findBuildings(ICity city) {
		ArrayList<IBuilding> result = new ArrayList<>();
		for (IBuilding building : city.getBuildings()) {
			if (building.getOwner()==this){
				result.add(building);
			}
		}
		return result;
	}

	@Override
	public Optional<ITradingOffice> findTradingOffice(ICity city) {
		for (IBuilding building : city.getBuildings()) {
			if (building.getOwner()==this && building instanceof ITradingOffice){
				return Optional.of((ITradingOffice) building);
			}
		}
		return Optional.empty();
	}

	@Override
	public <T extends IBuilding> List<T> findBuildings(ICity city,Class<T> buildingClass) {
		List<T> result = new ArrayList<>();
		for (IBuilding building : city.getBuildings()) {
			ClassChecker checker = new ClassChecker(building.getClass());
			if (checker.extendsClass(buildingClass) || checker.implementsInterface(buildingClass)){
				if (building.getOwner().equals(this)) {
					result.add((T) building);
				}
			}
		}
		return result;
	}

	/**
	 * Marry a spouse
	 * @param spouse to be married
	 */
	public void marry(ISpouseData spouse){
		this.spouse = Optional.of(spouse);
	}

	@Override
	public void spouseDies() {
		spouse = Optional.empty();
	}

	@Override
	public IPersonalData getPersonalData() {
		return personalData;
	}

	@Override
	public Optional<ISpouseData> getSpouseData() {
		return spouse;
	}

	@Override
	public void updateRank(ESocialRank rank) {
		this.rank=rank;
	}

	@Override
	public List<INavigableVessel> findShips(ICity city) {
		List<INavigableVessel> ships = newArrayList();
		for (INavigableVessel ship : fleet) {
			if (ship.getLocation().equals(city.getCoordinates())) {
				ships.add(ship);
			}
		}
		return ships;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		Player player = (Player) o;

		return uuid.equals(player.uuid);

	}

	@Override
	public int hashCode() {
		return uuid.hashCode();
	}

	@Override
    public void receiveSum(long amount) {
		if (this instanceof IHumanPlayer) {
			getCompany().updateCash(amount);
		} else {
			getCompany().updateCashDirectly(amount);

		}
    }

	@Override
	public void updateCrimialDrive(int value) {
		criminalDrive += value;
	}
}
