package ch.sahits.game.openpatrician.model.map;

import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import com.google.common.collect.Multimap;
import javafx.geometry.Dimension2D;
import javafx.geometry.Point2D;

import java.util.List;
import java.util.Optional;
import java.util.Set;


/**
 * Interface defining the map that defines the game plan
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 9, 2011
 *
 */
public interface IMap {
	/**
	 * Retrieve the number of cities that are on the map.
	 * @return number of cities on the map.
	 */
	int getNumberCities();
	/**
	 * Retrieve an unmodifiable list of all the cities in the map
	 * @return list of cities on the map, visible by all players.
	 */
	List<ICity> getCities();
    /**
     * Retrieve an unmodifiable list of all the cities in the map also including the once only visible to the player.
	 * @param player for whom the cities are visible.
     * @return list of cities on the map, inculding the ones only visible to <code>player</code>
     */
    List<ICity> getCities(IPlayer player);
    /**
     * Find a city by its name.
     * @param cityName name of the city
     * @return city reference.
     */
    ICity findCity(String cityName);

    /**
     * Retrieve the map dimensions
     * @return dimesion
     */
    Dimension2D getDimension();

    /**
     * Retrieve the path for the map image without the cities added.
     * @return original map image path.
     */
    String getMapImagePath();

	/**
	 * Retrieve the path for the black and white variant of the map.
	 * @return image file path for the black and white map for land and sea.
     */
	String getMapBWImagePath();

	/**
	 * Retrieve the future towns.
	 * @return list of future towns, that can be founded.
	 */
    List<ch.sahits.game.openpatrician.model.city.IFutureTowns> getFutureTowns();

	/**
	 * Retrieve the land bridges.
	 * @return list of landbridges that can be built.
	 */
    List<ILandBridge> getLandbridges();

	/**
	 * Retrieve the pirate nests
	 * @return list of pirate nests.
	 */
    List<ch.sahits.game.openpatrician.model.sea.PirateNest> getPirateNests();

	/**
	 * Retrieve the scale of the map by providing a number to convert one kilometer into
	 * a number of pixels.
	 * @return number of pixels that make up one km on the map.
     */
	double getNumberOfPixelPerKilometer();

	/**
	 * Find the city at the given coordinates.
	 * @param location of the supposed city.
	 * @return instance of the city or null if at the coordinates there is no city.
     */
	Optional<ICity> findCity(Point2D location);

	/**
	 * Retrieve a list of all coordinates of the cities.
	 * @return List of coordinates for all cities.
     */
	List<Point2D> getCityCoordinates();

	/**
	 * Retrieve the cities that are currently being founded by players.
	 * @return multimap of player who are founding cities.
     */
	Multimap<IPlayer, ICity> getTownsInFounding();

	/**
	 * Add a new city.
	 * @param city to be added
     */
	void add(ICity city);

	/**
	 * Retrieve the trade routes defined for this map.
	 * @return set of trade routes defined for this map.
     */
	Set<ITradeRoute> getTradeRoutes();
}
