package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.people.IWeaponsDealer;
import ch.sahits.game.openpatrician.model.weapon.ArmoryRegistry;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.model.weapon.IArmory;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.spring.DependentPropertyInitializer;
import ch.sahits.game.openpatrician.utilities.spring.DependentValue;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.property.ReadOnlyIntegerProperty;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
/**
 * Implementation of the weapons dealer.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 25, 2013
 *
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class WeaponsDealerState implements IWeaponsDealer {
	@XStreamOmitField
	private static final Logger LOGGER = LogManager.getLogger(WeaponsDealerState.class);
	@Autowired
	@XStreamOmitField
	private DependentPropertyInitializer propertyInitializer;
	@DependentValue("tavern.weapons.dealer.sellvalue.factor")
	private double sellValue;
	@Autowired
	private ArmoryRegistry armoryRegistry;

	private IArmory armory;

	@PostConstruct
	public void init() {
		try {
			propertyInitializer.initializeAnnotatedFields(this);
		} catch (IllegalAccessException e) {
			LOGGER.warn("Failed to initialize DependentValue annotated fields");
		}
	}

	/**
	 * Move ware into this holding. The player may be null and is not used in this
	 * base implementation, but subclasses may be interested for statistical reasons.
	 * This method is thread save.
	 * @param weapon to be moved
	 * @param amount of the ware that is moved
	 * @return the effective amount that was moved. The amount may be positive if something was added, negative if
	 * the ware was removed from the holding or zero if nothing was moved.
	 */
	@Override
	public int move(IWeapon weapon, int amount) {
		synchronized (weapon) { // There is still a problem with networked solution where the model of the city is replicated.
			int amounted = getWeapon(weapon);
			if (amount<0 && containsWeapon(weapon) &&  -amount>amounted){ // remove more than is available
				amount = -amounted; // remove everything
			}
			if (amount<0 && !containsWeapon(weapon)){
				amount = 0;
			}
			switch ((EWeapon)weapon) {
				case BALLISTA_SMALL:
					armory.updateBallistaSmallAmount(amount);
					break;
				case HAND_WEAPON:
					armory.updateCutlassAmount(amount);
					break;
				case TREBUCHET_SMALL:
					armory.updateTrebuchetSmallAmount(amount);
					break;
				case CANNON:
					armory.updateCanonAmount(amount);
					break;
				case BALLISTA_BIG:
					armory.updateBallistaBigAmount(amount);
					break;
				case TREBUCHET_BIG:
					armory.updateTrebuchetSmallAmount(amount);
					break;
				case BOMBARD:
					armory.updateBombardAmount(amount);
					break;
				case SWORD:
					armory.updateSwordAmount(amount);
					break;
				case BOW:
					armory.updateBowAmount(amount);
					break;
				case CROSSBOW:
					armory.updateCrossbowAmount(amount);
					break;
				case MUSKET:
					armory.updateMusketAmount(amount);
					break;
				default:
					throw new IllegalArgumentException("Unhandled weapon "+weapon);
			}
			return amount;
		}
	}
	/**
	 * Check if there is an amountable object for this ware
	 * @param weapons to be checked
	 * @return true if the weapon is contained.
	 */
	protected final boolean containsWeapon(IWeapon weapons) {
		return getWeapon(weapons) > 0;
	}

    /**
     * Retrieve the amount of the weapon that belongs to the dealer.
     * @param weapon type of weapon
     * @return amount available at the time of the call of the method.
     */
	private int getWeapon(IWeapon weapon) {
		return amountAvailableProperty((EWeapon)weapon).get();
	}


    @Override
    public ReadOnlyIntegerProperty amountAvailableProperty(EWeapon weapon) {
		switch (weapon) {
			case BALLISTA_SMALL:
				return armory.ballistaSmallAmountProperty();
			case HAND_WEAPON:
				return armory.cutlassAmountProperty();
			case TREBUCHET_SMALL:
				return armory.trebuchetSmallAmountProperty();
			case CANNON:
				return armory.canonAmountProperty();
			case BALLISTA_BIG:
				return armory.ballistaBigAmountProperty();
			case TREBUCHET_BIG:
				return armory.trebuchetBigAmountProperty();
			case BOMBARD:
				return armory.bombardAmountProperty();
			case SWORD:
				return armory.swordAmountProperty();
			case BOW:
				return armory.bowAmountProperty();
			case CROSSBOW:
				return armory.crossbowAmountProperty();
			case MUSKET:
				return armory.musketAmountProperty();
			default:
				throw new IllegalArgumentException("Unhandled weapon "+weapon);
		}    }

	@Override
	public void setCity(ICity city) {
		armory = armoryRegistry.getArmory(city);
	}
}
