package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.model.BaseAmountable;
import ch.sahits.game.openpatrician.model.IAmountable;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.base.Preconditions;
import javafx.beans.binding.NumberBinding;
import javafx.beans.binding.When;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleDoubleProperty;
/**
 * This class defines the amounted price of an {@link IWare} object.
 * Each object in the real world has hits unique price. Since no instances of the
 * objects are used but only one reference the price information gets lost. The
 * amountable price tracks the average price of all amounted objects
 * @param <T> {@link IAmountable} implementation that is is collected hereby called item
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 20, 2011
 *
 */
@Prototype
@ClassCategory({EClassCategory.PROTOTYPE_BEAN, EClassCategory.SERIALIZABLE_BEAN})
public class AmountablePrice<T extends IAmountable> extends BaseAmountable<T> {
	/** Total price of all amounted items */
	private final DoubleProperty sum = new SimpleDoubleProperty(this, "sum", 0);

    public AmountablePrice() {
    }

	/**
	 * Constructor initalizing the amountablePrice with an amount and average price.
	 * This constructor is mainly inteded for testing to avoid the hassle with the add method.
	 * @param amount represented by this.
	 * @param totalPrice total price of the complete amount
     */
	public AmountablePrice(int amount, double totalPrice) {
		Preconditions.checkArgument(amount >= 0, "Amount may not be negative");
		Preconditions.checkArgument(totalPrice >= 0, "Total price may not be negative");
		this.setAmount(amount);
		this.sum.setValue(totalPrice);
	}

    /**
	 * Retrieve the average price of one item
	 * @return average price per item
	 */
	public int getAVGPrice(){
		return (int)Math.rint(sum.doubleValue()/getAmount());
	}
	
	public NumberBinding avgPriceProperty() {
	    int amount = getAmount();
		return new When(sum.isEqualTo(0)
				        .or(amountProperty().isEqualTo(0)))
		          .then(0)
		          .otherwise(sum.divide(amount));
	}
	
	
	/**
	 * Add a number items
	 * @param amount number of the items to be added
	 * @param avgPrice average price of one item that is added
	 */
	public void add(final int amount, final int avgPrice){
		Preconditions.checkArgument(amount >= 0, "Amount may not be negative: "+amount);
		Preconditions.checkArgument(avgPrice >= 0, "Average price may not be negative: "+avgPrice);
		fxTheadExecution.execute(() -> {
			int localAmount = amount;
			if (localAmount<0){
				localAmount = Math.abs(localAmount);
			}
			int current = this.getAmount();
			this.setAmount(current+localAmount);
			long sum = this.sum.longValue();
			this.sum.set(sum + localAmount*avgPrice);
		});
	}
	/**
	 * Remove a number of items
	 * @param amount of items to be removed
	 */
	public void remove (final int amount){
		Preconditions.checkArgument(amount >= 0, "Amount may not be negative: "+amount);
		fxTheadExecution.execute(() -> {
			int localAmount = amount;
			if (localAmount<0){
				localAmount = Math.abs(localAmount);
			}
			if (localAmount==Integer.MIN_VALUE){
				localAmount = Integer.MAX_VALUE;
			}
			if (this.getAmount()<localAmount){
				this.setAmount(0);
				sum.set(0);
			} else {
				long sum = this.sum.longValue();
				this.sum.set(sum - localAmount*getAVGPrice()); // sum must be updated first
				int current = this.getAmount();
				this.setAmount(current - localAmount);
			}
			if (this.getAmount()==0){
				sum.set(0);
			}
		});
	}
	
	@Override
	public void reset() {
		fxTheadExecution.execute(() -> {
			this.sum.set(0);
			super.reset();
		});

	}

	public String toString(){
		return getAmount()+"@"+getAVGPrice();
	}

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }

        AmountablePrice<?> that = (AmountablePrice<?>) o;

        return sum != null ? sum.get() == that.sum.get() : that.sum == null;

    }

    @Override
    public int hashCode() {
        int result = super.hashCode();
        result = 31 * result + (sum != null ? sum.hashCode() : 0);
        return result;
    }

    /**
     * Get the total cum of the amount.
     * @return total price of all items.
     */
    public double getSum() {
        return sum.get();
    }

}
