package ch.sahits.game.openpatrician.model.server;

import ch.sahits.game.openpatrician.data.xmlmodel.map.Location;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.plugin.IMapPluginLoader;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.oxm.jaxb.Jaxb2Marshaller;

import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Service profiding access to map details, packed in the game an also in the plugins.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Feb 14, 2016
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@LazySingleton
public class MapProviderService {

    private final Logger logger = LogManager.getLogger(getClass());


    @Autowired
    @Qualifier("jaxb2XmlModelMarshallerMap")
    private Jaxb2Marshaller mapMarshaller;
    @ListType(IMapPluginLoader.class)
    @Autowired(required = false)
    private List<IMapPluginLoader> mapPluginLoaders;
    /**
     * Retrieve the map data consisting of key value pairs, where the key is the text key with which the map
     * should be represented and the value the path to the map definition XML on the classpath.
     * @return map of the text key of the map and the xml map definition path.
     */
    public Map<String, String> getMaps() {
        LinkedHashMap<String, String> maps = new LinkedHashMap<>();
        maps.put("ch.sahits.game.openpatrician.display.scene.NewGameScene.standard", "/standartHanseaticMap.xml");
        verifyMapDimenstions("/standartHanseaticMap.xml");
        if (mapPluginLoaders != null) {
            for (IMapPluginLoader mapLoader : mapPluginLoaders) {
                verifyMapDimenstions(mapLoader.getMapDefinition());

                maps.put(mapLoader.getMapTextKey(), mapLoader.getMapDefinition());
            }
        }
        return maps;
    }

    private void verifyMapDimenstions(String mapDefinition) {
        ch.sahits.game.openpatrician.data.xmlmodel.map.Map map =
                (ch.sahits.game.openpatrician.data.xmlmodel.map.Map)mapMarshaller.unmarshal(getSourceFromFile(mapDefinition));
        int heigth = map.getDimension().getY();
        for (ch.sahits.game.openpatrician.data.xmlmodel.map.Map.Cities.City city : map.getCities().getCity()) {
            Location location = city.getLocation();
            if (location.getX() < 30) {
               logger.warn("The city {} must be at least 30 pixels from the left edge", city.getName());
            }
            if (location.getY() < 40) { // 20 pixels above text
                logger.warn("The city {} must be at least 40 pixels from the  top border", city.getName());
            }
            if (location.getY() > heigth - 92) {
                logger.warn("The city {} must be at lease 92 pixels from the bottom border", city.getName());
            }
        }
    }

    private Source getSourceFromFile(String fileName) {
        final InputStream resourceAsStream = getClass().getResourceAsStream(fileName);
        return new StreamSource(resourceAsStream);
    }
}
