package ch.sahits.game.openpatrician.model.ship;

import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.BooleanProperty;
import javafx.geometry.Point2D;

import java.util.Optional;
import java.util.Set;

/**
 * Na navigable vessel is either a ship or a group of ships that
 * sail together.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 24, 2015
 */
public interface INavigableVessel {
	/**
	 * Total capacity of the ship in barrels.
	 * @return total holding capacity of the vessel in barrels
	 */
	int getSize();

	/**
	 * Available capacity of the ship, this represent the free storage in barrels
	 * @return free capacity of the vessel in barrels
	 */
	int getCapacity();

	/**
	 * Retrieve the name of the of the ship
	 * @return name of the vessel.
	 */
	String getName();

	/**
	 * Retrieve a list of all loaded wares
	 * @return set of loaded wares.
	 */
	Set<IWare> getLoadedWares();

	/**
	 * Load the ware in the specified amount onto the ship.
	 * The amount uses the size specified by the ware
	 * @param ware to be loaded
	 * @param amount of items the ware
	 * @param avgPrice average price of one item of the ware
	 * @return amount actually loaded
	 */
	int load(IWare ware, int amount, int avgPrice);

	/**
	 * Unload the specified quantity of the ware.
	 * The amount uses the size specified by the ware
	 * @param ware to be unloaded
	 * @param amount amount of items of ware to unload
	 * @return amount of the ware unloaded
	 */
	int unload(IWare ware, int amount);

	/**
	 * Retrieve the damage of the ship in percentage [0,100]
	 * @return damage of the vessel.
	 */
	int getDamage();

	/**
	 * Damage the ship
	 * @param damage amount of damage that is inflicted
	 * @param destroyWeapon flag indicating if weapons may be destroyed.
	 */
	void damage(int damage, boolean destroyWeapon);

	/**
	 * Retrieve the amount of load in barrels that is loaded in actual wares (no wappen or crew)
	 * @return amount of loaded wares in barrels
	 */
	IntegerBinding getLoadBinding();

	/**
	 * Retrieve the amount of ware loaded
	 * @param ware for which to retrieve price and amount
	 * @return price and amount loaded on the vessel.
	 */
	AmountablePrice<IWare> getWare(IWare ware);

	/**
	 * Retrieve the number of sailors on the ship.
	 * @return number of sailors on the vessel
	 */
	int getNumberOfSailors();

	/**
	 * Retrieve the captain of the ship.
	 * There might not be a captain present.
	 * @return Optional captain on the vessel. Empty if none is present.
	 */
	Optional<ICaptain> getCaptian();

	/**
	 * Retrieve the location of the ship.
	 * @return current location of the vessel.
	 */
	Point2D getLocation();

	/**
     * Set the location of the ship.
     * @param location of the ship
      */
	void setLocation(Point2D location);

	/**
     * Retrieve the owner of the ship. The owner cannot be retrieved if the pirate flag is set.
     * @return owner of the vessel.
     */
	IShipOwner getOwner();

	/**
	 * Retrieve the amount of weapons on the ship
	 * @param weaponType SWORD | BOW | CROSSBOW | MUSKET
	 * @return amount available on the ship
	 */
	int getWeaponAmount(IWeapon weaponType);

	/**
	 * Retrieve the total amount of space that is available to store stuff.
	 * This is the total amount of space minus any space due to upgrades.
	 * @return avalable space in barrels
     */
	int getLoadableSpace();

	/**
	 * Retrieve the speed of the vessel in kilometer per hour
 	 * @return speed of the vessel in kilometer per hour.
     */
	double getTopSpeed();
	/**
	 * Current top speed of the ship.
	 * @return speed of the vessel in kilometer per hour.
	 */
	double getCurrentSpeed();

	/**
	 * Retrieve a universal unique identifier for the vessel.
	 * @return UUID as string.
	 */
	String getUuid();

	/**
	 * Indicate if the vessel has hissed the pirate flag (Jolly Roger)
	 * and the vessel therefore should be considered a pirate vessel.
	 * @return true if the Jolly Roger is hissed.
     */
	boolean getPirateFlag();

	/**
	 * Property defining the pirate activit y of the vessel.
	 * @return bolean property for the pirate flag.
	 */
	BooleanProperty pirateFlagProperty();

    /**
     * Hiss or lower the pirate flag (Jolly Roger).
     */
    void togglePirateFlag();
}
