package ch.sahits.game.openpatrician.model.ship.impl;

import ch.sahits.game.openpatrician.model.AmountableProvider;
import ch.sahits.game.openpatrician.model.javafx.bindings.LateIntegerBinding;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IPerson;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.PrimaryLargeWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.SecondaryLargeWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.ShipProperties;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.collections.NonReplacableMap;
import ch.sahits.game.openpatrician.utilities.service.LUIDProvider;
import ch.sahits.game.openpatrician.utilities.spring.DependentPropertyInitializer;
import ch.sahits.game.openpatrician.utilities.spring.DependentValue;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.geometry.Point2D;
import lombok.Getter;
import lombok.Setter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
@Prototype
public abstract class Ship implements IShip {
	@XStreamOmitField
	private final Logger logger = LogManager.getLogger(getClass());
	/** Amount of barrels space a sailor needs */
	@DependentValue("sailor.space.on.ship")
	private double spacePerSailor = 0.5;
	@DependentValue("passenger.space.on.ship")
	private int passengerSpace = 2;
	@Getter
	private int distanceInKmForOneHealthPointReduction;
	@Autowired
	private LUIDProvider luidProvider;
	@Autowired
	@XStreamOmitField
	private Random rnd;

	/** State of the design standard level */
    @Getter
	protected EShipUpgrade shipUpgradeLevel;
	/** Name of the ship */
    @Getter
    @Setter
	protected String name;
	/** Store the wares loaded on the ship together with their amount. The amount is ware specific*/
	@MapType(key=IWare.class, value = AmountablePrice.class)
	private Map<IWare,AmountablePrice<IWare>> loadedWare = new NonReplacableMap<>();
	@ObjectPropertyType(IShip.class)
	private ObjectProperty<IShip> parentShip = new SimpleObjectProperty<>(null);
	/**
	 * Binding representing the current load.
	 */
    @XStreamOmitField
	private LateIntegerBinding loadBinding = createLoadBinding();

	private double topSpeed;

	@Autowired
	@XStreamOmitField
	private AmountableProvider amountableProvider;
	@Autowired
	@XStreamOmitField
	private DependentPropertyInitializer propertyInitializer;
    @XStreamOmitField
	private BooleanBinding passengerPresent = new BooleanBinding() {
		@Override
		protected boolean computeValue() {
			return passenger.isPresent();
		}
	};

	private int loadedHandWeapons = 0;
	@Getter
    @Setter
	private List<IWeaponSlot> weaponSlots = new ArrayList<>();
    /** Initial value of the ship */
    private int initialValue;
	@Getter
    private int minNumberOfSailors;
    private int maxSailors;
    @Getter
	private Optional<ICaptain> captian = Optional.empty();
    @Getter
    private Optional<IPerson> passenger = Optional.empty();
    /**
	 * Fitness of the ship. This is reduced as it is damaged
	 */
	private int fitness = 100;

	private IntegerProperty nbSailors = new SimpleIntegerProperty(0);
    @Setter
    @Getter
	private Point2D location = null;
	private IShipOwner owner = null;
	@Setter
	@Getter
	private boolean available = true;
	/** Space occupied by wares not owned by the owner */
	@Getter
	@Setter
	private int occupiedSpace;
	/** Size in barrels */
	@Getter
	@Setter
	private int size;
	private int reductionLevel1;
	private int reductionLevel2;
	@Getter
	private String uuid;
	private BooleanProperty pirateFlag = new SimpleBooleanProperty(false);

	protected final void initializeProperties(ShipProperties.Ship shipConfiguration) {
		this.initialValue = shipConfiguration.getInitialValue();
		this.minNumberOfSailors = shipConfiguration.getMinNumberOfSailors();
		this.maxSailors = shipConfiguration.getMaxSailors();
		this.size = shipConfiguration.getSize();
		this.reductionLevel1 = shipConfiguration.getReductionLevel1();
		this.reductionLevel2 = shipConfiguration.getReductionLevel2();
		this.topSpeed = shipConfiguration.getSpeedkmperh();
		this.distanceInKmForOneHealthPointReduction = shipConfiguration.getDistanceInKmForOneHealthPointReduction();
	}

	@PostConstruct
	private void init() {
		if (uuid == null) { // avoid overriding when the UUID is loaded
			uuid = luidProvider.getNextLUID();
		}
		try {
			propertyInitializer.initializeAnnotatedFields(this);
		} catch (IllegalAccessException e) {
			logger.warn("Failed to initialize DependentValue annotated fields");
		}
	}


	@Override
	public int getMaxNumberOfSailors() {
		int availableSpace = getCapacity();
		int spaceForSailors = (int)Math.rint(availableSpace/ spacePerSailor);
		return Math.min(maxSailors, spaceForSailors);
	}


	@Override
	public void setCaptain(ICaptain captain) {
		this.captian = Optional.ofNullable(captain);
	}

	@Override
	public boolean isUpgradable() {
		return shipUpgradeLevel != EShipUpgrade.LEVEL2;
	}

	@Override
	public void upgrade() {
		EShipUpgrade[] levels = EShipUpgrade.values();
		for (int i = 0; i < levels.length; i++) {
			if (levels[i]==shipUpgradeLevel){
                shipUpgradeLevel = levels[i+1];
				break; // only one update allowed
			}
		}
	}


	@Override
	public Set<IWare> getLoadedWares() {
		return loadedWare.keySet();
	}

	/**
	 * {@inheritDoc}
	 * This method is not thread safe as it is only intended to be accessed by one thread at a time
	 */
	@Override
	public int load(IWare ware, int amount, int avgPrice) {
		final short sizeInBarrels = ware.getSizeAsBarrels();
		amount = Math.abs(amount);
		int cap = ensureCapacity(amount*sizeInBarrels);
		// Make sure that for wares in other sizes than barrels we only load
		// complete loads.
		if (cap!=amount && sizeInBarrels!=1){
			cap = (cap/sizeInBarrels)*sizeInBarrels;
		}
		AmountablePrice<IWare> available = getWare(ware);
		available.add(cap/ware.getSizeAsBarrels(), avgPrice);
		return cap/sizeInBarrels;
	}

	/**
	 * Check if the amount can be loaded
	 * @param amount in barrels that is tried to load
	 * @return maximum amount that can be loaded.
	 */
	private int ensureCapacity(int amount) {
		return Math.max(Math.min(amount, getCapacity()), 0);
	}

	/**
	 * {@inheritDoc}
	 * This method is not thread safe as it is only intended to be accessed by one thread at a time
	 */
	@Override
	public int unload(IWare ware, int amount) {
		if (!loadedWare.containsKey(ware)){
			return 0; // nothing to unload
		}
		amount = Math.abs(amount);
		// convert to barrels
		amount = amount*ware.getSizeAsBarrels();
		final int loaded = loadedWare.get(ware).getAmount()*ware.getSizeAsBarrels();
		int unloaded = Math.min(loaded, amount);
		if (unloaded==loaded){ // unloaded completely
			AmountablePrice<IWare> available = loadedWare.get(ware);
			available.reset();
		} else {
			AmountablePrice<IWare> available = loadedWare.get(ware);
			available.remove(unloaded/ware.getSizeAsBarrels());
		}
		return unloaded/ware.getSizeAsBarrels();
	}

	/**
	 * Calculate the amount that is loaded on the ship in barrels.
	 * @return amount of loaded wares in barrels.
	 */
	private int internalLoadCalculation() {
		int sum=0;
		for (Entry<IWare,AmountablePrice<IWare>> entry : loadedWare.entrySet()) {
			int amount = entry.getValue().getAmount();
			int barrelSize = entry.getKey().getSizeAsBarrels();
			sum += amount*barrelSize;
		}
		return sum;
	}
	@Override
	public IntegerBinding getLoadBinding() {
		if (loadBinding == null) {
			loadBinding = createLoadBinding();
		}
		return loadBinding;
	}

	/**
	 * Clear all loaded wares. This method is only intended for testing
	 */
	protected void clearLoadedWares() {
		for (AmountablePrice<IWare> amountable : loadedWare.values()) {
			amountable.reset();
		}
	}

	@Override
	public AmountablePrice<IWare> getWare(IWare ware) {
		if (!loadedWare.containsKey(ware)){
			AmountablePrice<IWare> amountable = amountableProvider.createWareAmountable();
			loadedWare.put(ware, amountable);
			((LateIntegerBinding)getLoadBinding()).bind(amountable.amountProperty());
		}
		return loadedWare.get(ware);
	}

	@Override
	public int getValue() {
		return (int)Math.rint(initialValue*fitness/100.0);
	}


	@Override
	public int getNumberOfSailors() {
		return nbSailors.get();
	}


	@Override
	public void setNumberOfSailors(int nbSailors) {
		this.nbSailors.set(nbSailors);
	}

    @Override
    public IntegerProperty numberOfSailorsProperty() {
        return nbSailors;
    }

	@Override
	public int getWeaponAmount(IWeapon weaponType) {
		if (EWeapon.HAND_WEAPON == weaponType) {
			return loadedHandWeapons;
		}
		return (int) weaponSlots.stream()
				.filter(slot -> slot.getWeapon().isPresent())
				.filter(slot -> slot.getWeapon().get() == weaponType)
				.count();
	}
	@Override
	public void addPassenger(IPerson passanger) {
		Preconditions.checkArgument(!passenger.isPresent(), "Passenger already present");
		passenger = Optional.of(passanger);
		passengerPresent.invalidate();
	}
	@Override
	public IShipOwner getOwner() {
			return owner;
	}

	@Override
	public void setOwner(IShipOwner owner) {
		Preconditions.checkNotNull(owner, "Owner of a ship may not be null");
		if (this.owner != null) {
			logger.warn("Set owner of ship " + getName() + " of " + this.owner.getName()+" "+ this.owner.getLastName() + " to " + owner.getName() + " " + owner.getLastName(), new Exception("Stacktrace"));
		}
		this.owner = owner;
	}

	/**
	 * Passenger leave ship.
	 */
	@Override
	public void leavePassenger(){
		passenger = Optional.empty();
		passengerPresent.invalidate();
	}

	@Override
	public int getCapacity() {
		int passengerSpace = passenger.map(iPerson -> this.passengerSpace).orElse(0);
		return size-getUpgradeSpaceReduction()-internalLoadCalculation()-getOccupiedSpace() - passengerSpace - getOccupiedSpaceByWeapons();
	}

	@Override
	public int getUpgradeSpaceReduction(){
		switch (shipUpgradeLevel) {
			case NONE:
				return 0;
			case LEVEL1:
				return reductionLevel1;
			case LEVEL2:
				return reductionLevel2;
			default:
				return 0;
		}
	}

	private LateIntegerBinding createLoadBinding() {
		return new LateIntegerBinding() {

			@Override
			protected int computeValue() {
				return internalLoadCalculation();
			}
		};
	}

    public BooleanBinding passengerPresentProperty() {
        return passengerPresent;
    }
	@Override
	public int getDamage() {
		return 100 - fitness;
	}

	@Override
	public void damage(int damage, boolean destroyWeapon) {
		Preconditions.checkArgument(damage > 0, "The damage must be positive");
		fitness = Math.max(0, fitness - damage);
		boolean destroyAWeapon = destroyWeapon && rnd.nextInt(fitness) < 50;
		if (destroyAWeapon && hasWeapons()) {
			int slotIndex = rnd.nextInt(weaponSlots.size());
			while (true) {
				IWeaponSlot slot = weaponSlots.get(slotIndex);
				if (slot.getWeapon().isPresent()) {
					EWeapon weapon = (EWeapon) slot.getWeapon().get();
					slot.setWeapon(null);
					if (slot instanceof SecondaryLargeWeaponSlot && isLargeWeapon(weapon)) {
						weaponSlots.get(slotIndex - 1).setWeapon(null);
					}
					if (slot instanceof PrimaryLargeWeaponSlot && isLargeWeapon(weapon)) {
						weaponSlots.get(slotIndex + 1).setWeapon(null);
					}
					break;
				}
			}
		}
	}
	private boolean hasWeapons() {
		for (IWeaponSlot weaponSlot : weaponSlots) {
			if (weaponSlot.getWeapon().isPresent()) {
				return true;
			}
		}
		return false;
	}

	private boolean isLargeWeapon(EWeapon weapon) {
		switch (weapon) {
			case TREBUCHET_BIG:
			case BALLISTA_BIG:
			case BOMBARD:
				return true;
			default:
				return false;
		}
	}

	@Override
	public void repair() {
		fitness = 100;
	}

	@Override
    public int getLoadableSpace() {
        return getSize() - getUpgradeSpaceReduction();
    }

    @Override
    public double getTopSpeed() {
        return topSpeed;
    }

	@Override
	public boolean getPirateFlag() {
		return pirateFlag.get();
	}

	@Override
	public void togglePirateFlag() {
		pirateFlag.setValue(pirateFlag.get());
	}

	@Override
	public BooleanProperty pirateFlagProperty() {
		return pirateFlag;
	}

	@Override
	public double getCurrentSpeed() { // TODO: andi 9/30/17 move this to a property or binding 
		if (getDamage() == 0) {
			return topSpeed;
		}
		return Math.max(0.5, topSpeed / getDamage()/ 100.0);
	}

	@Override
	public ObjectProperty<IShip> parentShipProperty() {
		return parentShip;
	}
	@Override
	public void updateHandweapon(int delta) {
		loadedHandWeapons += delta;
	}

	@Override
	public int getOccupiedSpaceByWeapons() {
		double sum = getWeaponSlots().stream()
				.filter(slot -> slot.getWeapon().isPresent())
				.mapToDouble(slot -> slot.getWeapon().get().getSize())
				.sum();
		return (int) Math.rint(sum);
	}
}