package ch.sahits.game.openpatrician.model.city;

import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.city.impl.CityState;
import ch.sahits.game.openpatrician.model.city.impl.Contributions;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.geometry.Point2D;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Model of a city.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 16, 2011
 *
 */
public interface ICity {
	/**
	 * Retrieve the unique ID of the city.
	 * @return UUID of the city
	 */
	String getUniqueID();
	/**
	 * Retrieve the city type
	 * @return kontor type of the city
	 */
	EKontorType getKontorType();
	/**
	 * Retrieve the city name
	 * @return localized name of the city.
	 */
	String getName();

	/**
	 * Retrieve the current population of the city by class
	 * @param popclass for which to retrieve the population
	 * @return population of the relevant <code>popclass</code>
	 */
	int getPopulation(EPopulationClass popclass);
	/**
	 * Set the population by class
	 * @param population size of the population of the specified <code>popclass</code>
	 * @param popClass population class for which to set the population.
	 */
	void setPopulation(int population, EPopulationClass popClass);
	/**
	 * Add or remove ware from the city. If the amount is negative it will be removed.
	 * If the full amount is not available only the available will be moved and returned.
	 * The amount is in the ware specific size.
	 * @param ware to be moved
	 * @param amount of the ware that is moved
	 * @param player that moved the ware
	 * @return amount moved
	 */
	int move(IWare ware, int amount, ICitizen player);
	/**
	 * Retrieve the wares that are produced inefficiently
	 * @return array of wares produced ineffectively.
	 */
	IWare[] getIneffectiveProduction();
	/**
	 * Retrieve the wares that are produced efficiently
	 * @return array of wares produced effectively.
	 */
	IWare[] getEffectiveProduction();
	/**
	 * Retrieve a list of all buildings in the city. The list is immutable
	 * @return list of buildings in the city.
	 */
	List<IBuilding> getBuildings();
	/**
	 * Add a new building to the city
	 * @param building building to be removed.
	 */
	void tearDown(IBuilding building);
	/**
	 * Remove a building from the list of buildings in the city
	 * @param building to be added as completed
	 */
	void build(IBuilding building);
	/**
	 * Retrieve the reputation of the player in the city
	 * @param player for whom to retrieve the reputation
	 * @return reputation of a player
	 */
	IReputation getReputation(IPlayer player);

	/**
	 * Add a player to the city
	 * @param player that is added to the city
	 * @param reputaion of that player in the city
	 * @param contributions contibutions of that player in the city.
	 */
	void moveIn(IPlayer player, IReputation reputaion, Contributions contributions);

	/**
	 * Check out what the player contributed to the ware
	 * @param player whose contribution is to be checked
	 * @param ware that is checked for contribution
	 * @return amount of the contribution may also be negative
	 */
	int getContribution(IPlayer player, IWare ware);
	/**
	 * Retrieve the missing wares along with their times
	 * @return map of wares and since when they are missing
	 */
	Map<IWare, LocalDateTime> getMissingWares();
	/**
	 * Retrieve the coordinates of the city
	 * @return coordinates of the city
	 */
	Point2D getCoordinates();
	/**
	 * Retrieve all the players that have a office in this city.
	 * @return immutable list
	 */
	List<IPlayer> getResidentPlayers();
	/**
	 * Retrieve the city state.
	 * @return Retrieve hte city state
	 */
	CityState getCityState(); // TODO aho Feb 3, 2013: create interface

	/**
	 * Set the city state
	 * @param state state of the city.
	 */
	void setCityState(CityState state);
	/**
	 * Retrieve the amount of ware in the city
	 * @param ware for which to retrieve amount and price
	 * @return amount and price of the ware.
	 */
	AmountablePrice<IWare> getWare(IWare ware);
	/**
	 * Retrieve the population count for a class
	 * @param popclass population class
	 * @return integer property for the population of a specific <code>popclass</code>
	 */
	IntegerProperty getPopulationProperty(EPopulationClass popclass);
	/**
	 * Retrieve the binding of the city's total population.
	 * @return Integer binding of the overall population.
	 */
	IntegerBinding getPopulationBinding();

    /**
     * Retrieve the list of citizen in the city. This list contains various player as well as some instances used by the
     * AI to govern the city.
     * @return List of citizens in the city.
     */
    List<ICitizen> getCitizen();

    /**
     * Find all matching buildings in a city.
     * @param buildingClass class of the building
     * @param owner search can be restricted to a player.
     * @return List of buildings of type T
     */
    <T extends IBuilding> List<T> findBuilding(Class<T> buildingClass, Optional<IPlayer> owner);

	/**
	 * Retrieve the non locale sepecific name template.
	 * @return non localized name template
	 */
	String getNameTemplate();

	/**
	 * Check if the city is on a river instead of the open sea.
	 * @return true if the city is located on a river (instead of sea)
     */
	boolean isRiverCity();

	/**
	 * Retrieve the subset list of wares that are produced but are actually imported.
	 * @return List of wares that are imported.
	 */
	List<IWare> getImported();

	/**
	 * The player has building permission in this city.
	 * @param player for whom to check building permission
	 * @return true if <code>player</code> has building permission.
	 */
	boolean hasBuildingPermission(IPlayer player);

	/**
	 * Add the bulidng permission for that player in the city.
	 * @param player for whom to add a building permission
	 */
	void addBuildingPermission(IPlayer player);

	/**
	 * Retrieve the amount of road in the city. The amount is the value of number road tiles versus the
	 * population.
	 * @return percentage [0,1] of the built road in the city.
	 */
	double getPercentageRoad();

}
