package ch.sahits.game.openpatrician.model.city.guild.impl;

import ch.sahits.game.openpatrician.model.IDateService;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.guild.IMediterreanCity;
import ch.sahits.game.openpatrician.model.city.guild.IMediterreanMap;
import ch.sahits.game.openpatrician.model.city.guild.ITradeAgreement;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.MultimapType;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import javafx.geometry.Point2D;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Nov 06, 2016
 */
@LazySingleton
@ClassCategory({EClassCategory.SINGLETON_BEAN, EClassCategory.SERIALIZABLE_BEAN})
public class MediterreanSeaMap implements IMediterreanMap {
    private final int DETECTION_RADIUS = 10;
    @MultimapType(key = IPlayer.class, value = Point2D.class)
    private Multimap<IPlayer, Point2D> exploredSpots = ArrayListMultimap.create();
    @MapType(key = Point2D.class, value = String.class)
    private Map<Point2D, IMediterreanCity> cities = new HashMap<>();
    @ListType(Point2D.class)
    private List<Point2D> discoveredTradingSpots = new ArrayList<>();
    @MultimapType(key = IPlayer.class, value = ITradeAgreement.class)
    private Multimap<IPlayer, ITradeAgreement> tradeAgreements = ArrayListMultimap.create();
    @Autowired
    private IDateService dateService;

    @Override
    public boolean exploreSpot(IPlayer player, Point2D coordinates) {
        exploredSpots.put(player, coordinates);
        for (Point2D point : cities.keySet()) {
            if (point.distance(coordinates) <= DETECTION_RADIUS) {
                if (!discoveredTradingSpots.contains(point)) {
                    discoveredTradingSpots.add(point);
                    return true;
                }
                break;
            }
        }
        return false;
    }
    @Override
    public Map<Point2D, IMediterreanCity> getDiscoveredTradingSpots() {
        Map<Point2D, IMediterreanCity> discovered = new HashMap<>();
        for (Point2D point : discoveredTradingSpots) {
            discovered.put(point, cities.get(point));
        }
        return discovered;
    }

    @Override
    public List<Point2D> getExploredLocations(IPlayer player) {
        return new ArrayList<>(exploredSpots.get(player));
    }
    @VisibleForTesting
    void addCity(Point2D location, IMediterreanCity city) {
        cities.put(location, city);
    }

    @Override
    public void addTradeAgreement(IPlayer player, ITradeAgreement agreement) {
        cleanupTradeAgreements();
        tradeAgreements.put(player, agreement);
    }

    @Override
    public List<ITradeAgreement> getTradeAgreements(IPlayer player) {
        cleanupTradeAgreements();
        return new ArrayList<>(tradeAgreements.get(player));
    }

    private void cleanupTradeAgreements() {
        for (IPlayer player : tradeAgreements.keys()) {
            for (ITradeAgreement agreement : tradeAgreements.get(player)) {
                LocalDateTime validity = agreement.getValidTill();
                if (dateService.isPast(validity)) {
                    tradeAgreements.remove(player, agreement);
                }
            }
        }
    }
}
