package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import lombok.Getter;

import java.util.ArrayList;
import java.util.List;

/**
 * Container to keep track of state that is required for the population update
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Nov 05, 2017
 */
@Getter
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class PopulationUpdateStats {

    private int nbHiredSailors = 0;
    private boolean feedThePoor = false;
    private int hiredWorkers = 0;
    @ListType(INavigableVessel.class)
    private List<INavigableVessel> arrivedShips = new ArrayList<>();
    @ListType(INavigableVessel.class)
    private List<INavigableVessel> departedShips = new ArrayList<>();
    /**
     * Reset the state.
     */
    public void reset(){
          nbHiredSailors = 0;
          feedThePoor = false;
          hiredWorkers = 0;
          arrivedShips.removeAll(departedShips); // the rest is still in the city
          departedShips.clear();
    }

    /**
     * Amount of sailors hired.
     * @param nbSailors number of sailors that were hired
     */
    public void hireSailors(int nbSailors) {
        nbHiredSailors += nbSailors;
    }

    /**
     * Indicate that the poor were feed.
     */
    public void feedingThePoor() {
        feedThePoor = true;
    }

    /**
     * Number of workers hired in workshops
     * @param nbWorkers number of workers that were hired
     */
    public void hiredInWorkshop(int nbWorkers) {
         hiredWorkers += nbWorkers;
    }

    /**
     * Arriving ships
     * @param vessel that arrived
     */
    public void arrives(INavigableVessel vessel) {
        arrivedShips.add(vessel);
    }

    /**
     * Departing ships.
     * @param vessel that departed
     */
    public void depart(INavigableVessel vessel) {
        departedShips.add(vessel);
    }
}
