package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import com.google.common.base.Preconditions;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleDoubleProperty;

/**
 * This amountable keeps track of the amounts in a more detailed fashion than
 * {@link AmountablePrice}. It is only intended for internal use.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Oct 08, 2017
 */
@ClassCategory(EClassCategory.MODEL)
public class AmountablePriceDouble {

    private final DoubleProperty amount = new SimpleDoubleProperty(0);
    private final DoubleProperty sum = new SimpleDoubleProperty(0);

    public double getAmount() {
        return amount.get();
    }
    /**
     * Retrieve the average price of one item
     * @return average price of one item.
     */
    public int getAVGPrice(){
        return (int)Math.rint(sum.doubleValue()/getAmount());
    }

    public double getAVGPriceExact(){
        return sum.doubleValue()/getAmount();
    }
    /**
     * Add a number items
     * @param amount number of the items to be added
     * @param avgPrice average price of one item that is added
     */
    public void add(final double amount, final double avgPrice){
        Preconditions.checkArgument(amount >= 0, "Amount may not be negative: "+amount);
        Preconditions.checkArgument(avgPrice >= 0, "Average price may not be negative: "+avgPrice);
        double localAmount = amount;
        if (localAmount<0){
            localAmount = Math.abs(localAmount);
        }
        double current = this.getAmount();
        this.amount.setValue(current+localAmount);
        long sum = this.sum.longValue();
        this.sum.set(sum + localAmount*avgPrice);
    }

    /**
     * Remove a number of items
     * @param amount of items to be removed
     */
    public void remove (final double amount){
        Preconditions.checkArgument(amount >= 0, "Amount may not be negative: "+amount);
        double localAmount = amount;
        if (localAmount<0){
            localAmount = Math.abs(localAmount);
        }
        if (localAmount==Double.MIN_VALUE){
            localAmount = Double.MAX_VALUE;
        }
        if (this.getAmount()<localAmount){
            this.amount.setValue(0);
            sum.set(0);
        } else {
            double sum = this.sum.doubleValue();
            this.sum.set(sum - localAmount*getAVGPriceExact()); // sum must be updated first
            double current = this.getAmount();
            this.amount.setValue(current - localAmount);
        }
        if (this.getAmount()==0){
            sum.set(0);
        }
    }
}
