package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.model.city.ECityState;
import ch.sahits.game.openpatrician.model.city.IPopulationStructure;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MultimapType;
import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import javafx.geometry.Point2D;

import java.util.ArrayList;
import java.util.List;

/**
 * Price calculation if the curve is defined by a start and end point and two control points,
 * defining the curve as a bezier curve.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Aug 14, 2016
 */
@ClassCategory({EClassCategory.HANDLER, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class BezierPriceCalculation extends BasePriceCalulation {
    @MultimapType(key = ITradable.class, value = Point2D.class)
    private Multimap<ITradable, Point2D> cacheBuy = Multimaps.synchronizedMultimap(ArrayListMultimap.create());
    @MultimapType(key = ITradable.class, value = Point2D.class)
    private Multimap<ITradable, Point2D> cacheSell = Multimaps.synchronizedMultimap(ArrayListMultimap.create());
    /**
     * Calculate the point (x,y-coordinates) on the bezier curve defined by the four control points for
     * parameter <code>t</code>
     * @param t Value in the range [0,1] defining the fragment length on the curve.
     * @param start control point marking the start of the curve
     * @param cp1 first control point defining the tangential direction of the curve leaving <code>start</code>
     * @param cp2 second control point defining the tangential direction of the curve going into <code>end</code>
     * @param end control point marking the end of the curve.
     * @return Point (x,y-coordinates) at position <code>t</code> on the curve.
     */
    public Point2D caclulateBezierPoint(double t, Point2D start, Point2D cp1, Point2D cp2, Point2D end) {
        // B(t) = (1-t)³*start + 3*(1-t)²*t*cp1 + 3*(1-t)*t²*cp2 + t³*end
        double u = 1 - t;
        double tSquare = t*t;
        double tQube = t*t*t;
        double uSquare = u*u;
        double uQube = u*u*u;

        Point2D summand1 = start.multiply(uQube); // (1-t)³*start
        Point2D summand2 = cp1.multiply(3 * uSquare * t); // 3*(1-t)²*t*cp1
        Point2D summand3 = cp2.multiply(3 * u * tSquare); // 3*(1-t)*t²*cp2
        Point2D summand4 = end.multiply(tQube); // t³*end
        return summand1.add(summand2).add(summand3).add(summand4);
    }

    /**
     * Calculate a list of points at discrete x coordinates.<br>
     * PRE: The bezier curve does not bend back on itself.
     * @param start control point marking the start of the curve
     * @param cp1 first control point defining the tangential direction of the curve leaving <code>start</code>
     * @param cp2 second control point defining the tangential direction of the curve going into <code>end</code>
     * @param end control point marking the end of the curve.
     * @return List of points for x coordinates from start to end.
     */
    public List<Point2D> calculateDiscreteBezierPoints(Point2D start, Point2D cp1, Point2D cp2, Point2D end) {
        Preconditions.checkArgument(end.getX() > start.getX(), "The x coordinate of the end point must be larger than the one of the start point");
        int nbPoints = (int) Math.rint(end.getX() - start.getX());
        double delta = 0.015/nbPoints; // step size of t
        double epsilon = 0.1; // stop criterium when found point's x value is assumed accurate enough
        List<Point2D> points = new ArrayList<>(nbPoints);
        double lastX = start.getX() - 1;
        double t;
        for (t = 0; t <= 1; t += delta) {
            Point2D nextPoint = caclulateBezierPoint(t, start, cp1, cp2, end);
            if (nextPoint.getX() > lastX + 0.5) {
                double descreteX = (int)Math.rint(nextPoint.getX());
                if (Math.abs(descreteX - nextPoint.getX()) < epsilon) { // if the distance between the next integer and x is smaller than epsilon
                    lastX = nextPoint.getX();
                    Point2D p = new Point2D(Math.rint(nextPoint.getX()), Math.rint(nextPoint.getY()));
                    points.add(p);
                }
            }
        }
        if (points.size() == nbPoints) { // last point is missing
            points.add(end);
        }
        return points;
    }

    @Override
    public int computePrice(ITradable tradable, boolean buy, int available, int productionRate, IPopulationStructure pop, ECityState state) {
        int min = getMinValue(tradable, buy);
        int max = getMaxValue(tradable, buy);
        int saturation = getSaturation(tradable, buy);
        if (available==0) return max;
        if (available>=saturation) return min;
        if (buy) {
            synchronized (cacheBuy) {
                if (!cacheBuy.containsKey(tradable)) {
                    IBezierPriceCurve curve = tradable.getBuyCurve();
                    List<Point2D> points = calculateDiscreteBezierPoints(curve.getStart(), curve.getControlPoint1(), curve.getControlPoint2(), curve.getEnd());
                    cacheBuy.putAll(tradable, points);
                }
            }
        } else {
            synchronized (cacheSell) {
                if (!cacheSell.containsKey(tradable)) {
                    IBezierPriceCurve curve = tradable.getSellCurve();
                    List<Point2D> points = calculateDiscreteBezierPoints(curve.getStart(), curve.getControlPoint1(), curve.getControlPoint2(), curve.getEnd());
                    cacheSell.putAll(tradable, points);
                }
            }
        }

        return getPrice(buy, tradable, available);
    }

    private int getPrice(boolean buy, ITradable tradable, int available) {
        List<Point2D> points;
        if (buy) {
            points = new ArrayList<>(cacheBuy.get(tradable));
        } else {
            points = new ArrayList<>(cacheSell.get(tradable));
        }
        int index =  points.size() - 1;
        if (available >= 0) {
            index = Math.min(available, points.size() - 1);
        }
        Point2D firstGuess = points.get(index);
        if ((int)firstGuess.getX() == index) {
            return (int) firstGuess.getY();
        } else if (firstGuess.getX() > index) {
            for (int i = index - 1; i >= 0 ; i--) {
                Point2D p = points.get(i);
                if ((int)p.getX() == index) {
                    return (int) p.getY();
                }
            }
        } else {
            for (int i = index + 1; i < points.size(); i++) {
                Point2D p = points.get(i);
                if ((int)p.getX() == index) {
                    return (int) p.getY();
                }
            }
        }
        return -1;
    }
}
