package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

/**
 * Model for the production chain. All values refer to one workshop with peak efficiency and efficient production.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on May 14, 2016
 */
@Service
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class ProductionChain {

    private Properties properties = new Properties();
    @PostConstruct
    void init() throws IOException {
        String propertyFileName = "productionchain.properties";
        InputStream inputStream = getClass().getClassLoader().getResourceAsStream(propertyFileName);

        if (inputStream != null) {
            properties.load(inputStream);
        } else {
            throw new FileNotFoundException("property file '" + propertyFileName + "' not found in the classpath");
        }
    }

    public double getProduction(IWare ware) {
         String key = "productionchain.amount." + ware.name();
        return Double.parseDouble(properties.getProperty(key));
    }

    public List<IWare> getRequiredWares(IWare wareToBeProduced) {
        String key = "productionchain." + wareToBeProduced.name();
        if (properties.containsKey(key)) {
            String value = properties.getProperty(key);
            String[] wares = value.split(",");
            ArrayList<IWare> wareList = new ArrayList<>();
            for (String ware : wares) {
                wareList.add(EWare.valueOf(ware));
            }
            return wareList;
        } else {
            return Collections.emptyList();
        }
    }

    /**
     * Get the required amount of <code>requiredWare</code> to produce <code>wareToBeProduced</code> within a week.
     * @param wareToBeProduced ware that should be produced
     * @param requiredWare ware that is required
     * @return amount of <code>requiredWare</code> to produce <code>wareToBeProduced</code> within a week.
     */
    public double getRequiredAmount(IWare wareToBeProduced, IWare requiredWare) {
        String key = "productionchain." + wareToBeProduced.name() + ".amount." + requiredWare.name();
        if (properties.containsKey(key)) {
            return Double.parseDouble(properties.getProperty(key));
        } else {
            return 0;
        }
    }

    /**
     * Retrieve all wares that require other wares to be produced.
     * @return List of wares that are produced in a production chain.
     */
    public List<IWare> getWaresWithProductionChain() {
         List<IWare> list = new ArrayList<>();
        for (EWare ware : EWare.values()) {
            if (!getRequiredWares(ware).isEmpty()) {
                list.add(ware);
            }
        }
        return list;
    }
}
