package ch.sahits.game.openpatrician.model.sea;

import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.ship.EShipTravelState;
import ch.sahits.game.openpatrician.model.ship.IGroupableVessel;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import javafx.geometry.Point2D;
import javafx.scene.shape.Path;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * This iterable collection stores all vessels that are currently travelling.
 * Vessels should be added when they start their travel and be removed when they reach their destination or
 * otherwise stop traveling (e.g. are sunk).
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jan 08, 2016
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
@Component
@Lazy
@Slf4j
public class TravellingVessels implements ITravellingVessels {
    @MapType(key = INavigableVessel.class, value = TravellingVessel.class)
    private final Map<INavigableVessel, TravellingVessel> vessels = new ConcurrentHashMap<>();
    @Autowired
    private IMap map;

    /**
     * The <code>vessel</code> starts it's travel and must be added to this collection together with its data.
     * @param vessel that starts the travel
     * @param path the Bezière path representing the route.
     * @param points the list of points that make up the path along which the <code>vessel</code> is travelling.
     */
    public void addVessel(INavigableVessel vessel, Optional<Path> path, List<Point2D> points) {
        Point2D destination = points.get(points.size() - 1);
         Optional<ICity> destCity = findCity(destination);
        if (vessel instanceof IShip) {
            setTravelDestionation((IShip) vessel, destCity);
        } else if (vessel instanceof IGroupableVessel) {
            for (IShip ship : ((IGroupableVessel) vessel).getShips()) {
                setTravelDestionation(ship, destCity);
            }
        }
        TravellingVessel tv = new TravellingVessel(vessel);
        tv.setCalculatablePath(points);
        path.ifPresent(tv::setDrwawablePath);
        vessels.put(vessel, tv);
        log.debug("Add vessel {} ({}) to travelling vessels.", vessel.getName(), vessel.getUuid());
    }

    private void setTravelDestionation(IShip vessel, Optional<ICity> destCity) {
        if (destCity.isPresent()) {
            vessel.setTravelState(EShipTravelState.TRAVEL_TO_CITY);
        } else {
            vessel.setTravelState(EShipTravelState.TRAVEL_TO_DESTINATION);
        }
    }

    /**
     * The vessel is no longer travelling and should no longer be part of this collection.
     * @param vessel to be removed
     */
    public void remove(INavigableVessel vessel) {
        vessels.remove(vessel);
        log.debug("Remove vessel {} ({}) frome travelling vessels.", vessel.getName(), vessel.getUuid());
    }

    @Override
    public Iterator<INavigableVessel> iterator() {
        return vessels.keySet().iterator();
    }

    @Override
    public TravellingVessel getTravellingVessel(INavigableVessel vessel) {
        return vessels.get(vessel);
    }

    @Override
    public boolean isTravelling(INavigableVessel vessel) {
        return vessels.containsKey(vessel);
    }

    /**
     * Find the city that is located at the postion <code>location</code>.
     * @param location that should be checked.
     * @return Optional of a ICity if there is a city at <code>location</code>.
     */
    public Optional<ICity> findCity(Point2D location) {
        for (ICity city : map.getCities()) {
            if (city.getCoordinates().equals(location)) {
                return Optional.of(city);
            }
        }
        return Optional.empty();
    }
}
