package ch.sahits.game.openpatrician.model.service.persistance.converter;

import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.converters.collections.AbstractCollectionConverter;
import com.thoughtworks.xstream.io.ExtendedHierarchicalStreamWriterHelper;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.mapper.Mapper;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Stream;

/**
 * Converter for Maps. General Maps are delegated to the MapConverter while any map of type
 * EWare, Number are handled by this converter.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jun 26, 2016
 */
@Slf4j
public class WareMapConverter extends AbstractCollectionConverter {

    public static final String INT_ENTRY_NAME = "wareIntMapEntry";
    public static final String DOUBLE_ENTRY_NAME = "wareDoubleMapEntry";
    public static final String AMOUNTABLEPRICE_ENTRY_NAME = "wareAmountablePriceMapEntry";

    public WareMapConverter(Mapper mapper) {
        super(mapper);
    }

    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
        try {
            Map map = (Map) source;
            if (map.isEmpty()) {
                marshall(writer, context, map);
            } else {
                Object firstKey = map.keySet().iterator().next();
                Object firstValue = map.get(firstKey);
                if (firstKey instanceof EWare && (firstValue instanceof Integer || firstValue instanceof  Double || firstValue instanceof AmountablePrice)) {
                   if (firstValue instanceof Integer) {
                       for (Object o : map.keySet()) {
                           EWare ware = (EWare) o;
                           int value = (Integer) map.get(ware);
                           marshallMapEntry(writer, INT_ENTRY_NAME, ware.name(), String.valueOf(value));
                       }
                   } else if (firstValue instanceof  Double){
                       for (Object o : map.keySet()) {
                           EWare ware = (EWare) o;
                           double value = (Double) map.get(ware);
                           marshallMapEntry(writer, DOUBLE_ENTRY_NAME, ware.name(), String.valueOf(value));
                       }
                   } else {
                       for (Object o : map.keySet()) {
                           EWare ware = (EWare) o;
                           AmountablePrice value = (AmountablePrice) map.get(ware);
                           marshallAmountablePriceMapEntry(writer, AMOUNTABLEPRICE_ENTRY_NAME, ware.name(), value.getAmount(), value.getSum());
                       }
                   }
                } else {
                    marshall(writer, context, map);
                }
            }
        } catch (RuntimeException e) {
            try {
                Field f = Stream.of(context.getClass().getDeclaredFields()).filter(field -> field.getName().endsWith("currentPath")).findFirst().get();
                f.setAccessible(true);
                String path = f.get(context).toString();
                log.warn("Failed to convert {} for the savegame on path {}", source.getClass().getName(), path);
            } catch (IllegalAccessException ex) {
                log.warn("Failed to retrieve path from context", ex);
            }
            throw e;
        }
    }

    private void marshallAmountablePriceMapEntry(HierarchicalStreamWriter writer, String entryName, String name, int amount, double avgPrice) {
        writer.startNode(entryName);
        writer.addAttribute("ware", name);
        writer.addAttribute("amount", String.valueOf(amount));
        writer.addAttribute("avgPrice", String.valueOf(avgPrice));
        writer.endNode();
    }

    private void marshall(HierarchicalStreamWriter writer, MarshallingContext context, Map map) {
        String entryName = mapper().serializedClass(Entry.class);
        for (Object o : map.entrySet()) {
            Entry entry = (Entry) o;
            marschallMapEntry(writer, context, entryName, entry);
        }
    }

    private void marschallMapEntry(HierarchicalStreamWriter writer, MarshallingContext context, String entryName, Entry entry) {
        ExtendedHierarchicalStreamWriterHelper.startNode(writer, entryName, entry.getClass());

        writeItem(entry.getKey(), context, writer);
        writeItem(entry.getValue(), context, writer);

        writer.endNode();
    }


    private void marshallMapEntry(HierarchicalStreamWriter writer, String entryName, String name, String value3) {
        writer.startNode(entryName);
        writer.addAttribute("ware", name);
        writer.addAttribute("value", value3);
        writer.endNode();
    }

    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
        Map map = (Map) createCollection(context.getRequiredType());
        populateMap(reader, context, map);
        return map;
    }

    private void populateMap(HierarchicalStreamReader reader, UnmarshallingContext context, Map map) {
        populateMap(reader, context, map, map);
    }
    private void populateMap(HierarchicalStreamReader reader, UnmarshallingContext context, Map map, Map target) {
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            String nodeName = reader.getNodeName();
            switch (nodeName) {
                case INT_ENTRY_NAME:
                case DOUBLE_ENTRY_NAME:
                    putCurrentEntryIntoMap(reader, context, map, target, nodeName);
                    break;
                case AMOUNTABLEPRICE_ENTRY_NAME:
                    putCurrentAmountablePriceEntryIntoMap(reader, context, map, target, nodeName);
                    break;
                default:
                    putCurrentEntryIntoMap(reader, context, map, target);
                    break;
            }
            reader.moveUp();
        }
    }

    private void putCurrentAmountablePriceEntryIntoMap(HierarchicalStreamReader reader, UnmarshallingContext context, Map map, Map target, String nodeName) {
        EWare key = EWare.valueOf(reader.getAttribute("ware"));
        int amount = Integer.parseInt(reader.getAttribute("amount"));
        double avgPrice = Double.parseDouble(reader.getAttribute("avgPrice"));
        AmountablePrice<IWare> amountablePrice = new AmountablePrice<>(amount, avgPrice);
        target.put(key, amountablePrice);
    }

    private void putCurrentEntryIntoMap(HierarchicalStreamReader reader, UnmarshallingContext context, Map map, Map target, String nodeName) {
        EWare key = EWare.valueOf(reader.getAttribute("ware"));
        Number value;
        if (nodeName.equals(INT_ENTRY_NAME)) {
            value = Integer.parseInt(reader.getAttribute("value"));
        } else {
            value = Double.parseDouble(reader.getAttribute("value"));
        }
        target.put(key, value);

    }

    private void putCurrentEntryIntoMap(HierarchicalStreamReader reader, UnmarshallingContext context,
                                          Map map, Map target) {
        reader.moveDown();

        Object key = readItem(reader, context, map);
        reader.moveUp();

        reader.moveDown();
        Object value = readItem(reader, context, map);
        reader.moveUp();
        target.put(key, value);
    }


    public boolean canConvert(Class type) {
        return type.equals(HashMap.class)
                || type.equals(Hashtable.class)
                || type.getName().equals("java.util.LinkedHashMap")
                || type.getName().equals("java.util.concurrent.ConcurrentHashMap")
                || type.getName().equals("sun.font.AttributeMap"); // Used by java.awt.Font in JDK 6
    }
}
