package ch.sahits.game.openpatrician.model.ship.impl;

import ch.sahits.game.openpatrician.model.AmountableProvider;
import ch.sahits.game.openpatrician.model.javafx.bindings.LateIntegerBinding;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.ICog;
import ch.sahits.game.openpatrician.model.ship.ICrayer;
import ch.sahits.game.openpatrician.model.ship.IHolk;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IShipAutoTrading;
import ch.sahits.game.openpatrician.model.ship.IShipGroup;
import ch.sahits.game.openpatrician.model.ship.ISnaikka;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.geometry.Point2D;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.Set;

/**
 * Implementation of a group of ships mainly used by pirates.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Apr 10, 2016
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class ShipGroup implements IShipGroup {
    @Autowired
    @XStreamOmitField
    private Random rnd;
    private BooleanProperty pirateFlag = new SimpleBooleanProperty(false);
    @ListType(IShip.class)
    private List<IShip> ships = new ArrayList<>();
    @Autowired
    @XStreamOmitField
    private AmountableProvider amountableProvider;
    public ShipGroup(IShip leadShip) {
        ships.add(leadShip);
    }
    /**
     * Binding representing the current load.
     */
    @XStreamOmitField
    private LateIntegerBinding loadBinding = null;

    @Override
    public int getSize() {
        return ships.stream().mapToInt(INavigableVessel::getSize).sum();
    }

    @Override
    public int getCapacity() {
        return ships.stream().mapToInt(INavigableVessel::getCapacity).sum();
    }

    @Override
    public String getName() {
        return ships.isEmpty() ? "" : ships.get(0).getName();
    }

    @Override
    public Set<IWare> getLoadedWares() {
        Set<IWare> loadedWares = new HashSet<>();
        for (IShip ship : ships) {
            loadedWares.addAll(ship.getLoadedWares());
        }
        return loadedWares;
    }

    @Override
    public int load(IWare ware, int amount, int avgPrice) {
        int loadedAmount = 0;
        int remainingAmount = amount;
        for (IShip ship : ships) {
            if (ship.getCapacity() > 0) {
                loadedAmount += ship.load(ware, remainingAmount, avgPrice);
                remainingAmount = amount - loadedAmount;
                if (remainingAmount <= 0) {
                    break;
                }
            }
        }
        return loadedAmount;
    }

    @Override
    public int unload(IWare ware, int amount) {
        int unloadedAmount = 0;
        int remainingAmount = amount;
        for (IShip ship : ships) {
            unloadedAmount += ship.unload(ware, remainingAmount);
            remainingAmount = amount - unloadedAmount;
            if (remainingAmount <= 0) {
                break;
            }
        }
        return unloadedAmount;
    }

    @Override
    public int getDamage() {
        return ships.stream().mapToInt(INavigableVessel::getDamage).max().getAsInt();
    }

    @Override
    public void damage(int damage, boolean destroyWeapon) {
        for (IShip ship : ships) {
            double rndFactor = Math.max(rnd.nextGaussian() + 1, 0);
            double typeFactor = 1;
            if (ship instanceof IHolk) {
                typeFactor = 0.75;
            }
            if (ship instanceof ICog) {
                typeFactor = 1;
            }
            if (ship instanceof ICrayer) {
                typeFactor = 1.3;
            }
            if (ship instanceof ISnaikka) {
                typeFactor = 1.7;
            }
            int actualDamage = (int) Math.round(damage*rndFactor*typeFactor);
            ship.damage(actualDamage, destroyWeapon);
        }
    }

    private LateIntegerBinding loadbinding() {
        if (loadBinding == null) {
            loadBinding = createLoadBinding();
        }
        return loadBinding;
    }
    private LateIntegerBinding createLoadBinding() {
        return new LateIntegerBinding() {
            {
                for (IShip ship : ships) {
                    super.bind(ship.getLoadBinding());
                }
            }

            @Override
            protected int computeValue() {
                return ships.stream().mapToInt(ship -> ship.getLoadBinding().get()).sum();
            }

        };
    }
    @Override
    public IntegerBinding getLoadBinding() {
        return loadbinding();
    }

    @Override
    public AmountablePrice<IWare> getWare(IWare ware) {
        AmountablePrice<IWare> loadedWare = amountableProvider.createWareAmountable();
        for (IShip ship : ships) {
            AmountablePrice<IWare> amountable = ship.getWare(ware);
            loadedWare.add(amountable.getAmount(), amountable.getAVGPrice());
        }
        return loadedWare;
    }

    @Override
    public int getNumberOfSailors() {
        return ships.stream().mapToInt(INavigableVessel::getNumberOfSailors).sum();
    }

    @Override
    public int getMinNumberOfSailors() {
        return ships.stream().mapToInt(INavigableVessel::getMinNumberOfSailors).sum();
    }
    @Override
    public Optional<ICaptain> getCaptian() {
        return Optional.empty();
    }

    @Override
    public Point2D getLocation() {
        return ships.get(0).getLocation();
    }

    @Override
    public void setLocation(Point2D location) {
        for (IShip ship : ships) {
            ship.setLocation(location);
        }
    }

    @Override
    public IShipOwner getOwner() {
        return ships.get(0).getOwner();
    }

    @Override
    public int getWeaponAmount(IWeapon weaponType) {
        return ships.stream().mapToInt(ship -> ship.getWeaponAmount(weaponType)).sum();
    }

    @Override
    public int getLoadableSpace() {
        return ships.stream().mapToInt(INavigableVessel::getLoadableSpace).sum();
    }

    @Override
    public double getTopSpeed() {
        return ships.stream().mapToDouble(INavigableVessel::getTopSpeed).max().getAsDouble();
    }

    @Override
    public double getCurrentSpeed() {
        return ships.stream().mapToDouble(INavigableVessel::getCurrentSpeed).min().getAsDouble();
    }

    @Override
    public String getUuid() {
        return ships.get(0).getUuid();
    }

    @Override
    public boolean getPirateFlag() {
        return pirateFlag.get();
    }

    @Override
    public void togglePirateFlag() {
        pirateFlag.setValue(pirateFlag.get());
    }

    @Override
    public BooleanProperty pirateFlagProperty() {
        return pirateFlag;
    }

    @Override
    public List<IShip> getShips() {
        return new ArrayList<>(ships);
    }

    @Override
    public void addShip(IShip ship) {
         ships.add(ship);
    }

    @Override
    public void removeShip(IShip ship) {
        ships.remove(ship);
    }

    /**
     * {@inheritDoc}
     * A ship group does not support autotrading.
     * @return empty
     */
    @Override
    public Optional<IShipAutoTrading> getAutotrading() {
        return Optional.empty();
    }

    @Override
    public void setAutoTrading(IShipAutoTrading autoTrading) {

    }
}
