package ch.sahits.game.openpatrician.model.building;

import ch.sahits.game.openpatrician.model.IBalanceSheet;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;

import java.time.LocalDateTime;
import java.util.Optional;

/**
 * The trading office (Kontor) is the trading point in a city. You can buy/sell between the trading
 * office and the city. You can move wares and wares between your storage and any ship currently in
 * the city.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Dec 17, 2011
 *
 */
public interface ITradingOffice extends IBuilding {
	/**
	 * Retrieve the amount of ware in storage
	 * @param ware to be checked in the trading office.
	 * @return amount and price of the <code>ware</code> stored in the trading office.
	 */
	AmountablePrice<IWare> getWare(IWare ware);
	/**
	 * check if there are any weapons in storage
	 * @return true if weapons are stored.
	 */
	boolean hasWeapons();
	/**
	 * Add or remove ware from the storage. If the amount is negative it will be removed.
	 * If the full amount is not available only the available will be moved and returned.
	 * The amount is in the ware specific size.
	 * The price of the wares is computed based on the amount
	 * @param ware to be moved in/out of the trading office
	 * @param amount of <code>ware</code> to be moved.
	 * @return amount that is acctualy moved
	 * @deprecated Use {@link #move(IWare, int, int)}
	 */
	@Deprecated
	int move(IWare ware, int amount);
	/**
	 * Add or remove weapon from the storage. If the amount is negative it will be removed.
	 * If the full amount is not available only the available will be moved and returned.
	 * The amount is in the ware specific size.
	 * @param weapons weapon that should be moved
	 * @param amount of <code>weapons</code> that should be moved
	 * @return actual amount moved
	 */
	int move(IWeapon weapons, int amount);
	/**
	 * Retrieve the storage capacity of the trading office and all other storages in the same city
	 * @return storage capacity in barrels
	 */
	IntegerProperty capacityProperty();
	/**
	 * Move the ware into/from the storage with a price attached.
	 * @param ware to be moved
	 * @param moved amount to be moved. Negative values indicate removal of the <code>ware</code> from the
	 *                 trading office.
	 * @param avgPrice average price to use on the amount of ware
	 * @return the amount that was actually moved. If the <code>ware</code> was to be removed the returned value
	 * is &lt;= 0.
	 */
	int move(IWare ware, int moved, int avgPrice);
	/**
	 * Retrieve the balance sheet for last completed week.
	 * @return last weeks balance sheet
	 */
	IBalanceSheet getBalanceLastWeek();
	/**
	 * Retrieve the balance sheet for the current week
	 * @return current weeks balance sheet
	 */
	IBalanceSheet getCurrentWeek();
	/**
	 * Retrieve the steward of the trading office (manager). May be null if no manager is employed
	 * @return Optional steward. Empty if none is hired.
	 */
	Optional<ISteward> getSteward();
	/**
	 * Setting the steward. May be null to indicate firing the steward
	 * @param steward to set/unset (null)
	 */
	void setSteward(ISteward steward);
	/**
	 * Get the trading object of the office
	 * @return Automated trading instance for this office
	 */
	IAutomatedTrading getOfficeTrading();
	/**
	 * Retrieve the weapon storage for this trading office
	 * @return storage of the weapons
	 */
	IWeaponStorage getWeaponStorage();
	/**
	 * Retrieve an overview over the storage
	 * @return ware storage
	 */
	IStorage getStorage();
	/**
	 * Binding representing the current stored amount.
	 * @return integer binding of the stored amount in barrels.
	 */
	IntegerBinding storedAmountBinding();

    /**
     * Property holding the steward.
     * @return object property for the steward
     */
    ObjectProperty<ISteward> stewardProperty();

    /**
     * Binding indicating the presence of a steward.
     * @return boolean binding for the presence of a steward.
     */
    BooleanBinding storageManagerPresentBinding();

	/**
	 * Retrieve tha amount of occupied space not available to the player.
	 * @return occupied space in barrels
	 */
	int getOccupiedSpace();

	/**
	 * Set the amount of occupied space not available to the player.
	 * @param space new occupied space
	 */
	void setOccupiedSpace(int space);

	/**
	 * Retrieve the city the trading office is located in.
	 * @return city in which the trading office is located.
	 */
	ICity getCity();

	/**
	 * Replace the last weeks balance sheet with the current week and
	 * set the new balance sheet for this week.
	 * @param newWeek balance sheet to replace the old one.
     */
	void replaceBalanceSheet(IBalanceSheet newWeek);

	/**
	 * Retrieve the date when the trading office was established.
	 * @return date when the trading office was built.
	 */
	LocalDateTime getEstablishedDate();
}
