package ch.sahits.game.openpatrician.model.impl;

import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.building.IWeaponStorage;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.model.weapon.IArmory;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.base.Preconditions;
import javafx.beans.property.ReadOnlyIntegerProperty;
import lombok.Getter;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.HashMap;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 *         Created on Jul 27, 2017
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class Armory implements IArmory {
    private final static double EXPERIENCE_GAIN = 0.005;

    @Autowired
    private IWeaponStorage weaponStorage;
    @Autowired
    private Date date;
    @Autowired
    private IMap map;

    private final ICity city;
    @MapType(key = IWeapon.class, value = Boolean.class)
    private HashMap<IWeapon, Boolean> priority = new HashMap<>();
    /** Value between 0 and 1 detailing the experience of the ship yard crew (the higher the better) */
    @Getter
    private double experience;
    /** Date until which the blacksmith is covered with work. */
    private LocalDateTime occupied;

    public Armory(ICity city) {
        this.city = city;
    }

    @PostConstruct
    private void init() {
        for (EWeapon weapon : EWeapon.values()) {
            priority.put(weapon, false);
        }
        int mapWidth = (int) map.getDimension().getWidth();
        double eastFactor = city.getCoordinates().getX()/mapWidth/0.5;
        double startYearPart = Math.max((date.getStartYear()-1430)/100,0);
        experience = Math.max(startYearPart-eastFactor,0);
        Preconditions.checkArgument(experience>=0, "Experiance must be positive");
        occupied = date.getCurrentDate().minusDays(1);
    }

    @Override
    public ReadOnlyIntegerProperty swordAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.SWORD);
    }

    @Override
    public ReadOnlyIntegerProperty bowAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.BOW);
    }

    @Override
    public ReadOnlyIntegerProperty crossbowAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.CROSSBOW);
    }

    @Override
    public ReadOnlyIntegerProperty musketAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.MUSKET);
    }

    @Override
    public ReadOnlyIntegerProperty cutlassAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.HAND_WEAPON);
    }

    @Override
    public ReadOnlyIntegerProperty trebuchetSmallAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.TREBUCHET_SMALL);
    }

    @Override
    public ReadOnlyIntegerProperty trebuchetBigAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.TREBUCHET_BIG);
    }

    @Override
    public ReadOnlyIntegerProperty ballistaSmallAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.BALLISTA_SMALL);
    }

    @Override
    public ReadOnlyIntegerProperty ballistaBigAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.BALLISTA_BIG);
    }

    @Override
    public ReadOnlyIntegerProperty bombardAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.BOMBARD);
    }

    @Override
    public ReadOnlyIntegerProperty canonAmountProperty() {
        return weaponStorage.getWeaponProperty(EWeapon.CANNON);
    }

    @Override
    public boolean isSwordPriority() {
        return priority.get(EWeapon.SWORD);
    }

    @Override
    public boolean isBowPriority() {
        return priority.get(EWeapon.BOW);
    }

    @Override
    public boolean isCrossbowPriority() {
        return priority.get(EWeapon.CROSSBOW);
    }

    @Override
    public boolean isMusketPriority() {
        return priority.get(EWeapon.MUSKET);
    }

    @Override
    public void toggleSwordPriority() {
        priority.put(EWeapon.SWORD, !priority.get(EWeapon.SWORD));
    }

    @Override
    public void toggleBowPriority() {
        priority.put(EWeapon.BOW, !priority.get(EWeapon.BOW));
    }

    @Override
    public void toggleCrossbowPriority() {
        priority.put(EWeapon.CROSSBOW, !priority.get(EWeapon.CROSSBOW));
    }

    @Override
    public void toggleMusketPriority() {
        priority.put(EWeapon.MUSKET, !priority.get(EWeapon.MUSKET));
    }

    @Override
    public void updateSwordAmount(int delta) {
        weaponStorage.update(EWeapon.SWORD, delta);
    }


    @Override
    public void updateBowAmount(int delta) {
        weaponStorage.update(EWeapon.BOW, delta);
    }

    @Override
    public void updateCrossbowAmount(int delta) {
        weaponStorage.update(EWeapon.CROSSBOW, delta);
    }

    @Override
    public void updateMusketAmount(int delta) {
        weaponStorage.update(EWeapon.MUSKET, delta);
    }

    @Override
    public void updateCutlassAmount(int delta) {
        weaponStorage.update(EWeapon.HAND_WEAPON, delta);
    }

    @Override
    public void updateTrebuchetSmallAmount(int delta) {
        weaponStorage.update(EWeapon.TREBUCHET_SMALL, delta);
    }

    @Override
    public void updateTrebuchetBigAmount(int delta) {
        weaponStorage.update(EWeapon.TREBUCHET_BIG, delta);
    }

    @Override
    public void updateBallistaSmallAmount(int delta) {
        weaponStorage.update(EWeapon.BALLISTA_SMALL, delta);
    }

    @Override
    public void updateBallistaBigAmount(int delta) {
        weaponStorage.update(EWeapon.BALLISTA_BIG, delta);
    }

    @Override
    public void updateBombardAmount(int delta) {
        weaponStorage.update(EWeapon.BOMBARD, delta);
    }

    @Override
    public void updateCanonAmount(int delta) {
        weaponStorage.update(EWeapon.CANNON, delta);
    }

    @Override
    public void gainExperience() {
        experience += EXPERIENCE_GAIN;
    }

    @Override
    public LocalDateTime occupiedUntil() {
        return occupied;
    }

    @Override
    public void occupy(LocalDateTime date) {
       Preconditions.checkArgument(this.date.getCurrentDate().isBefore(date), "The date must be in the future");
       this.occupied = date;
    }
}
