package ch.sahits.game.openpatrician.model.people;

import ch.sahits.datastructure.GenericPair;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.map.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.product.IWare;

import java.util.Optional;

/**
 * Provide the information the informant is willing to sell.
 * The bits of information are optional, but one bit is always present.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 24, 2013
 *
 */
public interface IInformant extends ITavernPerson {
	/**
	 * Get the price of the information.
	 * @param player requesting the price
	 * @return price tag for the information
	 */
	int getPrice(IPlayer player);
	/**
	 * Retreive the name of the informant
	 * @return name of the informant.
	 */
	String getName();
	/**
	 * Possible information about in which city which ware is needed.
	 * @return Optional {@link GenericPair} of city and ware for missing wares.
	 */
	Optional<GenericPair<ICity, IWare>> getWareNeeded();
	/**
	 * Possible information about in which city which ware is available in surplus.
	 * @return Optional {@link GenericPair} of city and ware whith surplus of ware.
	 */
	Optional<GenericPair<ICity, IWare>> getWareSurplus();
	/**
	 * Possible information about the location of a pirate.
	 * @return Optional {@link GenericPair} of pirate name and city where the pirate is rumored to be.
	 */
	Optional<GenericPair<String, ICity>> getPirateLocation();

	/**
	 * Set the information about the needed ware in the city.
	 * @param city which needs ware
	 * @param ware that is needed
	 */
	void setNeededWare(ICity city, IWare ware);

	/**
	 * Set the information about the surplus ware in the city
	 * @param city which has surplus on wares
	 * @param ware that is available in abundance
	 */
	void setWareSurplus(ICity city, IWare ware);

	/**
	 * Set the location of the pirate.
	 * @param pirate for whom there is information available
	 * @param city near which the pirate has been seen.
	 */
	void setPirateLocation(ISeaPirate pirate, ICity city);
	/**
	 * The informant may also hold a map part for the player.
	 * @param player requesting player
	 * @return Optional of the {@link MapSegmentedImage}. Empty if there is no map segment for hte <code>player</code>
	 */
	Optional<MapSegmentedImage> getMapSectionPart(IPlayer player);

	/**
	 * Indicate if the informant has a piece of a map.
	 * @return true if the informant sells a piece of a map.
	 */
	boolean isShowMapIfAvailable();

	/**
	 * Set the flag for the map piece
	 * @param showMapIfAvailable true if the informant has a map piece
	 */
	void setShowMapIfAvailable(boolean showMapIfAvailable);

	/**
	 * Set the price of the information
	 * @param price of hte information
	 */
	void setPrice(int price);

	/**
	 * Set the name of the informant
	 * @param name of the informant
	 */
	void setName(String name);
}
