package ch.sahits.game.openpatrician.model;

import ch.sahits.game.openpatrician.model.player.AIPlayerContext;
import ch.sahits.game.openpatrician.model.player.IAIBuyWeaponStrategyType;
import ch.sahits.game.openpatrician.model.player.IAICaptainHireStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIConstructionSelectionStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIEventDecisionStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIGuildJoinStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIHireSailorsStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIHireTradeManagerStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIShipRepairStrategyType;
import ch.sahits.game.openpatrician.model.player.IAIShipUpgradeStrategyType;
import ch.sahits.game.openpatrician.model.player.IAITakeLoanStrategyType;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategyType;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;

import java.util.List;

/**
 * Player interface defining an artificial player.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 17, 2011
 *
 */
public interface IAIPlayer extends IPlayer {
    void setTradeStrategyType(INavigableVessel vessel, IAITradeStrategyType type);

    /**
     * Retreive the the type of trade strategy.
     * @param vessel for which to retrieve the trade strategy type
     * @return trade strategy type for the vessel
     */
    IAITradeStrategyType getTradeStrategyType(INavigableVessel vessel);

    /**
     * Retrieve the type of for the ship repair strategy.
     * @return ship repair strategy type.
     */
    IAIShipRepairStrategyType getShipRepairStrategyType();

    /**
     * Retrieve the type of the strategy for taking a loan.
     * @return strategy type to take out a loan
     */
    IAITakeLoanStrategyType getTakeLoanStrategyType();

    /**
     * Retrieve the strategy type for event decission.
     * @return strategy type to make event based decissions
     */
    IAIEventDecisionStrategyType getEventDecitionStrategyType();

    /**
     * Retrieve the type of the strategy to use for deciding how to build new ships.
     * @return strategy type for construction selection
     */
    IAIConstructionSelectionStrategyType getConstructionSelectionType();

    /**
     * Retrieve the type of the strategy to hire captains.
     * @return strategy type to hire captains
     */
    IAICaptainHireStrategyType getCaptainHireStrategyType();

    /**
     * Retrieve the strategy to join guilds.
     * @return strategy type to join guilds
     */
    IAIGuildJoinStrategyType getGuildJoinStrategyType();

    /**
     * Retrieve the strategy type for buying weapons.
     * @return strategy types to decide to buy weapons
     */
    IAIBuyWeaponStrategyType getWeaponBuyStrategyType();

    /**
     * Retrieve the strategy type for upgrading ships.
     * @return strategy type for ship upgrading
     */
    IAIShipUpgradeStrategyType getShipUpgradeStrategyType();

    /**
     * Retreive the strategy type for hiring the sailors.
     * @return strategy type to hire sailors
     */
    IAIHireSailorsStrategyType getHireSailorStrategyType();

    /**
     * Retrieve the strategy for hiring and dismissing trade managers.
     * @return strategy type to hire trade managers
     */
    IAIHireTradeManagerStrategyType getHireTradeManagerStrategyType();

    /**
     * Retrieve the knowledge of production and consumption.
     * @return global knowledge of production and consumption.
     */
    IProductionConsumptionKnowledge getProductionAndConsumptionKnowledge();

    /**
     * Retrieve the next trade step that should be executed and remove it from the list.
     * @param vessel for which the next trade step should be checked.
     * @return next trade step for the vessel
     */
    ITradeStep getNextTradeStep(INavigableVessel vessel);

    /**
     * Retrieve a copy of the trade steps of the vessel.
     * @param vessel for which the next trade step should be checked.
     * @return all trade steps for that vessel
     */
    List<ITradeStep> getTradeSteps(INavigableVessel vessel);

    /**
     * Check if there are further trade steps.
     * @param vessel which should be checked.
     * @return true if there are more trade steps defined for vessel
     */
    boolean hasMoreTradeSteps(INavigableVessel vessel);

    /**
     * Add a new trade step at the end of the list.
     * @param vessel for which the step should be added.
     * @param tradeStep to be added.
     */
    void addTradeStep(ITradeStep tradeStep, INavigableVessel vessel);

    /**
     * Retrieve the trade mission of a given vessel.
     * @param vessel for which to get the mission data
     * @return trade mission for a specific vessel
     */
    ITradeMissionData getTradeMission(INavigableVessel vessel);

    /**
     * Add trade mission data fo a vessel. If the data is null an existing entry will be removed.
     * @param vessel for which to set the trade mission data
     * @param tradeMission trade mission meta data
     */
    void setTradeMission(INavigableVessel vessel, ITradeMissionData tradeMission);

    /**
     * Inject a trade step at the beinning of the task list for the vessel
     * @param vessel for which the step should be added.
     * @param tradeStep to be added.
     */
    void injectTradeStep(ITradeStep tradeStep, INavigableVessel vessel);

    /**
     * Check if the next trade step can be executed for the vessel, or if the last
     * trade step requires waiting on an event.
     * @param vessel for which the status should be checked.
     * @return true if the next trade step is initiated upon an event.
     */
    boolean waitingForTradeStepToFinish(INavigableVessel vessel);

    /**
     * Update  the waiting status upon the execution of a trade step.
     * @param vessel for which the status should be updated.
     * @param wait inicate if the next trade step should wait.
     */
    void updateTradeWaitingStatus(INavigableVessel vessel, boolean wait);

    /**
     * Check if the trade strategy for the vessel is initialized.
     * @param vessel to be checked
     * @return true if the trade steps for the vessel are initialized.
     */
    boolean isInitialized(INavigableVessel vessel);

    /**
     * Retrieve the context for this player.
     * @return global context for the player.
     */
    AIPlayerContext getPlayerContext();

}
