package ch.sahits.game.openpatrician.model;

import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.map.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.personal.EEconomicCareer;
import ch.sahits.game.openpatrician.model.personal.EMilitantCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import ch.sahits.game.openpatrician.model.personal.IChild;
import ch.sahits.game.openpatrician.model.personal.IPersonalData;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;

import java.util.List;
import java.util.Optional;

/**
 * Base player in the game.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Jun 15, 2011
 *
 */
public interface IPlayer extends ICitizen, IShipOwner{
	/**
	 * Retrieve the personal data of the player
	 * @return personal data of the player.
	 */
	IPersonalData getPersonalData();
	/**
	 * Retrieve the personal data of the players spouse.
	 * @return Optional spouse data. Empty if the player is not married.
	 */
	Optional<ISpouseData> getSpouseData();
	/**
	 * Retrieve the company data
	 * @return company of the player.
	 */
	ICompany getCompany();

	/**
	 * Get a list of all ships
	 * @return list of the players ships.
	 */
	List<IShip> getFleet();

	/**
	 * Provide a different view on the fleet by containing all the
	 * vessels that can be selected.
	 * @return list of vessels that can be activated.
     */
	List<INavigableVessel> getSelectableVessels();

    /**
     * Add a selectable vessel to the list.
     * @param vessel to be added
     */
    void addSelectableVessel(INavigableVessel vessel);

    /**
     * Remove a selectable vessel from the list.
     * @param vessel to be removed.
     */
    void removeSelectableVessel(INavigableVessel vessel);
	/**
	 * Find all ships in a city
	 * @param city in which to fin ships
	 * @return list of vessels in the <code>city</code>
	 */
	List<INavigableVessel> findShips(ICity city);
	/**
	 * Add a new ship for the player
	 * @param ship to be added to the fleet.
	 */
	void addShip(IShip ship);
	/**
	 * Remove a ship for the player
	 * @param ship to be removed from the fleet.
	 */
	void removeShip(IShip ship);
	/**
	 * Find all buildings of the player in the city
	 * @param city where to search for buildings.
	 * @return list of buildings in <code>city</code> owned by the player.
	 */
	List<IBuilding> findBuildings(ICity city);
	/**
	 * Find the trading office of the player in the city.
	 * @param city where to look for a trading office
	 * @return Optional trading office. Empty if there is none in the <code>city</code>.
	 */
	Optional<ITradingOffice> findTradingOffice(ICity city);
	/**
	 * Find all buildings in the city that are some subtype of the indicated class
	 * @param city in which to look for buildings
	 * @param buildingClass class of the building that is searched
	 * @return type cast list of the buildings in <code>city</code>.
	 */
	<T extends IBuilding> List<T> findBuildings(ICity city, Class<T> buildingClass);

	/**
	 * Update to the new social rank
	 * @param rank new social rank.
	 */
	void updateRank(ESocialRank rank);
	/**
	 * Retrieve the career level, which is either defined by {@link EMilitantCareer} or {@link EEconomicCareer}.
	 * @return current career level.
	 */
	ICareer getCareerLevel();

    /**
     * Set the current career level.
     * @param career new career level
     */
	void setCareerLevel(ICareer career);

	/**
	 * Set or unset the image for a segmented map (treasure or pirate hideout).
	 * @param mapSegments to be set or unset (null)
	 */
	void setSegmentedMap(MapSegmentedImage mapSegments);

	/**
	 * Retrieve the segmented image map.
	 * @return segmented image map
	 */
	MapSegmentedImage getSegmentedMap();

	/**
	 * Conclude the marriage.
	 * @param spouse data for the spouse.
	 */
	void marry(ISpouseData spouse);

	/**
     * Widow and become unmarried again.
	 */
	void spouseDies();

    /**
     * Retrieve all children.
     * @return list of children
     */
    List<IChild> getChildren();

	/**
	 * Update the new value indicating the level of criminallity of the player
	 * @param value new criminal drive value.
	 */
	void updateCrimialDrive(int value);

	/**
	 * Retrieve the value indicating the criminal drive of the player.
	 * @return criminal drive
	 */
	int getCriminalDrive();

	/**
	 * Retrieve the UUID identifing this player.
 	 * @return UUID of the player.
	 */
	String getUuid();
}
