package ch.sahits.game.openpatrician.model.impl;

import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.javafx.IJavaFXApplicationThreadExecution;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.property.LongProperty;
import javafx.beans.property.SimpleLongProperty;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

@Slf4j
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public final class Company implements ICompany {

	private IPlayer owner = null;
    @Getter
	private ICity homeTown;
    @Getter
	@Setter
	private long companyValue;
	private LongProperty cash;

	@Autowired
	@XStreamOmitField
	private IJavaFXApplicationThreadExecution threadExecutor;


	public Company(ICity homeTown, long cash, long random) { // Random argument is to avoid returning duplicate beans when creating a new instance
		super();
		this.homeTown = homeTown;
		this.cash = new SimpleLongProperty(this, "cash", cash);
	}

	public void setOwner(IPlayer player) {
		Preconditions.checkNotNull(player);
		Preconditions.checkArgument(owner == null, "The owner may only be set once");
		this.owner = player;
	}


	@Override
	public long getCash() {
		return cash.get();
	}
	/**
	 * Add or subtract some cash on the UI thread
	 * @param diff amount of money that is transferred
	 */
	@Override
	public void updateCash(long diff){
		threadExecutor.execute(() ->
			updateCashDirectly(diff)
		);
	}

	/**
	 * Add or subtract some cash directly. This method is intended for calls
	 * that do not originate in the UI.
	 * @param diff delta that is to be updated (positive or negative)
	 */
	public void updateCashDirectly(long diff) {
		long oldValue = cash.get();
		cash.set(oldValue + diff);
		if (cash.get() < 0 && diff < 0) {
			log.warn("Company of {} {} (uuid {}) cash dropped below 0: {}", owner.getName(), owner.getLastName(), owner.getUuid(), cash.get());
		}
	}

	@Override
    public LongProperty cashProperty() {
        return cash;
    }
}
