package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.event.NewPirateEvent;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.people.INonFreeSeaPirate;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.utilities.annotation.SetType;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * State of all the sea pirates.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Feb 1, 2013
 *
 */
@LazySingleton
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
public class SeaPiratesState {

	@Autowired
	@XStreamOmitField
	private Random rnd;
	@Autowired
	private IMap map;
	@Autowired
	@Qualifier("serverClientEventBus")
	@XStreamOmitField
	private AsyncEventBus clientServerEventBus;

	@Autowired
	private Date date;

	@SetType(ISeaPirate.class)
	private Set<ISeaPirate> pirates =  Sets.newHashSet();
	/**
	 * Add a pirate. The pirate may be a private pirate belonging only to one player.
	 * Post an update event to notify listiners of a new pirate.
	 * @param pirate to be added
	 */
	public void add(ISeaPirate pirate) {
		pirates.add(pirate);
		clientServerEventBus.post(new NewPirateEvent(pirate));
	}

	/**
	 * Remove a pirate from the list.
	 * @param pirate to be removed
     */
	public void remove(ISeaPirate pirate) {
		pirates.remove(pirate);
	}

	/**
	 * A non free priate becomes a free pirate.
	 * @param pirate non free pirate who disapears
	 * @param replaceWith free pirate who appears
	 */
	public void replace(INonFreeSeaPirate pirate, ISeaPirate replaceWith) {
		pirates.remove(pirate);
		pirates.add(replaceWith);
	}
	/**
	 * Retrieve immutable list of non free pirates for the player
	 * @param player who has contracted non free pirates
	 * @return List of non free pirates contracted by <code>player</code>
	 */
	public List<ISeaPirate> getNonFreePirates(IPlayer player) {
		return Collections.unmodifiableList(pirates.stream()
				.filter(p -> p instanceof INonFreeSeaPirate && ((INonFreeSeaPirate)p).getHiringPlayer().equals(player))
				.collect(Collectors.toList()));
	}

	/**
	 * Retrieve all pirates.
	 * @return set of all pirates
     */
	public Set<ISeaPirate> getAllPirates() {
		return new HashSet<>(pirates);
	}
	/**
	 * Retrieve immutable list of free pirates.
	 * @return set of free sea pirates.
	 */
	public Set<ISeaPirate> getFreePirates(){
		return pirates.stream().filter(p -> !(p instanceof INonFreeSeaPirate)).collect(Collectors.toSet());
	}

	/**
	 * Find a successful free pirate
	 * @return sea pirate that is the most successful.
	 */
	public ISeaPirate findSuccessfulPirate() {
		Preconditions.checkArgument(!getFreePirates().isEmpty(), "There are no free pirates");
		int maxKill = 0;
		ISeaPirate successful = null;
		for (ISeaPirate pirate : getFreePirates()) {
			if (pirate.getNumberOfKills()>=maxKill) {
				successful = pirate;
				maxKill = pirate.getNumberOfKills();
			}
		}
		return successful;
	}
}
