package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.people.IBuyer;
import ch.sahits.game.openpatrician.model.people.IConcurrent;
import ch.sahits.game.openpatrician.model.people.IContractBroker;
import ch.sahits.game.openpatrician.model.people.ICourier;
import ch.sahits.game.openpatrician.model.people.IEscorte;
import ch.sahits.game.openpatrician.model.people.IFugitive;
import ch.sahits.game.openpatrician.model.people.IInformant;
import ch.sahits.game.openpatrician.model.people.IPatrol;
import ch.sahits.game.openpatrician.model.people.IPerson;
import ch.sahits.game.openpatrician.model.people.IPirate;
import ch.sahits.game.openpatrician.model.people.IPirateHunter;
import ch.sahits.game.openpatrician.model.people.ISailorState;
import ch.sahits.game.openpatrician.model.people.ISideRoomPerson;
import ch.sahits.game.openpatrician.model.people.ISmuggler;
import ch.sahits.game.openpatrician.model.people.ITavernPerson;
import ch.sahits.game.openpatrician.model.people.IThieve;
import ch.sahits.game.openpatrician.model.people.ITrader;
import ch.sahits.game.openpatrician.model.people.ITransportTrader;
import ch.sahits.game.openpatrician.model.people.ITraveler;
import ch.sahits.game.openpatrician.model.people.ITreasureMapOwner;
import ch.sahits.game.openpatrician.model.people.IWarehouseTenant;
import ch.sahits.game.openpatrician.model.people.IWeaponsDealer;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import ch.sahits.game.openpatrician.utilities.service.RandomNameLoader;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.binding.ObjectBinding;
import javafx.beans.property.MapProperty;
import javafx.beans.property.SimpleMapProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import lombok.Getter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;

/**
 * Each tavern engine has to store its state.
 * The tavern state is of type prototype as each city has its
 * separate tavern state, but there it is a singleton instance.
 * The initialisation of the tavern state happens in the city state.
 *
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Jan 19, 2013
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class TavernState {
    @Getter
    private String name;
    @Autowired
    @Getter
    private ITrader trader;
    @Autowired
    @Getter
    private IInformant informant;
    @Autowired
    @Getter
    private ITraveler traveler;
//    /**
//     * Brooker that handels the buying of wares that were agreed to
//     * be delivered through a notice in another city.
//     */
//    @Getter
//    @OptionalType(IContractBroker.class)
//    private Optional<IContractBroker> contractBrooker = Optional.empty();   // fixme: andi 3/20/15: the broker is player specific, there may be even more than one per player
    // Side room
    @Autowired
    @Getter
    private IThieve thieve;
    @Autowired
    @Getter
    private IEscorte escorte;
    @Autowired
    @Getter
    private IFugitive fugative;
    @Autowired
    @Getter
    private IPatrol patrol;
    @Autowired
    @Getter
    private ICourier courier;
    @Autowired
    @Getter
    private ITransportTrader transportTrader;
    @Autowired
    @Getter
    private IConcurrent concurrent;
    @Autowired
    @Getter
    private IBuyer buyer;
    @Autowired
    @Getter
    private IPirateHunter pirateHunter;
    @Autowired
    @Getter
    private IWarehouseTenant warehouseTenant;
    @Autowired
    @Getter
    private ISmuggler smuggler;
    @Autowired
    @Getter
    private ITreasureMapOwner treasureMap;
    @Autowired
    @Getter
    private IWeaponsDealer weaponsDealer;
    @Autowired
    @Getter
    private IPirate pirate;
    @Autowired
    @Getter
    private ISailorState sailors;
    @Autowired
    @Qualifier("resourceReference")
    @XStreamOmitField
    private MessageSource resources;
    @Autowired
    @XStreamOmitField
    private Locale locale;

    private ICity city;
    @ListType(ITavernPerson.class)
    private ObservableList<ITavernPerson> absentPersons;
    @ListType(IPerson.class)
    private ObservableList<IPerson> presentPersons;
    @XStreamOmitField
    private SideRoomBinding presentPersonsInSideRoom;
    @MapType(key=IPerson.class, value = IPerson.class)
    private MapProperty<IPerson, IPlayer> talkingTo;
    @ListType(ISideRoomPerson.class)
    private final List<ISideRoomPerson> sideRoomPersons = newArrayList();
    private RandomNameLoader tavernNameLoader;

    public TavernState(ICity city) {
        this.city = city;
        talkingTo = new SimpleMapProperty<>(this, "talkingTo");
        talkingTo.set(FXCollections.observableMap(new HashMap<>()));
    }
    public  ObservableList<ITavernPerson> absentPersonProperty() {
        return absentPersons;
    }
    public ObservableList<IPerson> presentPersonsProperty() {
        return presentPersons;
    }
    public ObjectBinding<List<ISideRoomPerson>> presentPersonsInSideRoomBinding() {
        if (presentPersonsInSideRoom == null) {
            presentPersonsInSideRoom = new SideRoomBinding();
        }
        return presentPersonsInSideRoom;
    }
    private void initCity() {

        buyer.setCity(city);
        concurrent.setCity(city);
        courier.setCity(city);
        escorte.setCity(city);
        fugative.setCity(city);
        informant.setCity(city);
        patrol.setCity(city);
        pirate.setCity(city);
        pirateHunter.setCity(city);
        smuggler.setCity(city);
        thieve.setCity(city);
        trader.setCity(city);
        transportTrader.setCity(city);
        traveler.setCity(city);
        treasureMap.setCity(city);
        warehouseTenant.setCity(city);
        weaponsDealer.setCity(city);
        name = tavernNameLoader.getRandomName();

        presentPersons.addAll(getPresentPersons()); // Has precondition city set

    }
    @PostConstruct
    private void initializePresenceBinding() {
        ObservableList<ITavernPerson> absentList  = FXCollections.observableList(getAbsentPersons());
        absentPersons = FXCollections.synchronizedObservableList(absentList);
        ObservableList<IPerson> syncBackingPresentList = FXCollections.observableArrayList();
        presentPersons = FXCollections.synchronizedObservableList(syncBackingPresentList);

        sideRoomPersons.add(courier);
        sideRoomPersons.add(transportTrader);
        sideRoomPersons.add(thieve);
        sideRoomPersons.add(escorte);
        sideRoomPersons.add(fugative);
        sideRoomPersons.add(patrol);
        sideRoomPersons.add(concurrent);
        sideRoomPersons.add(buyer);
        sideRoomPersons.add(pirateHunter);
        sideRoomPersons.add(warehouseTenant);
        sideRoomPersons.add(smuggler);
        sideRoomPersons.add(treasureMap);
        for (ISideRoomPerson sideRoomPerson : sideRoomPersons) {
              sideRoomPerson.isPresentProperty().addListener(new PresenceChangeListener(sideRoomPerson));
        }
        trader.isPresentProperty().addListener(new PresenceChangeListener(trader));
        informant.isPresentProperty().addListener(new PresenceChangeListener(informant));
        traveler.isPresentProperty().addListener(new PresenceChangeListener(traveler));
        pirate.isPresentProperty().addListener(new PresenceChangeListener(pirate));
        String resourceName = resources.getMessage("tavern.name", new Object[0], locale.getCurrentLocal());
        tavernNameLoader = new RandomNameLoader(resourceName);
        initCity();
    }

    private List<ITavernPerson> getAbsentPersons() {
        List<ITavernPerson> tavernPerson = asListAllAbsent();
        List<ITavernPerson> allAbsent = new ArrayList<>();
        if (!pirate.isPresent()) {
            allAbsent.add(pirate);
        }
        for (ITavernPerson aTavernPerson : tavernPerson) {
            allAbsent.add(aTavernPerson);
        }
        return Collections.synchronizedList(allAbsent);
    }

    /**
     * Retrieve a list of all persons present at the tavern
     * @return list of persons present in the tavern.
     */
    private List<IPerson> getPresentPersons() {
        Preconditions.checkNotNull(city);
        List<IPerson> tavernPerson = asListAllPresent();
        List<IPerson> allPresent = new ArrayList<>();
        int index = 0;
        if (!tavernPerson.isEmpty() && tavernPerson.get(index) instanceof IContractBroker) {
            allPresent.add(tavernPerson.get(index++));
        }
        allPresent.add(getSailors());
        if (pirate.isPresent()) {
            allPresent.add(pirate);
        }
        allPresent.add(getWeaponsDealer());
        for (int i = index; i < tavernPerson.size(); i++) {
            allPresent.add(tavernPerson.get(i));
        }
        return allPresent;
    }

    /**
     * Retrieve all people who are present
     * @return list of persons absent in the tavern
     */
    List<IPerson> asListAllPresent() {
        List<IPerson> list = new ArrayList<>();
        // contract broker is handled in the menu
        // nothing added for sailors
        // nothing added for captain
        // nothing added for weapons dealer
        // nothing added for pirate
        if (informant.isPresent()) {
            list.add(informant);
        }
        if (trader.isPresent()) {
            list.add(trader);
        }
        if (traveler.isPresent()) {
            list.add(traveler);
        }
        // side room
        if (thieve.isPresent()) {
            list.add(thieve);
        }
        if (escorte.isPresent()) {
            list.add(escorte);
        }
        if (fugative.isPresent()) {
            list.add(fugative);
        }
        if (patrol.isPresent()) {
            list.add(patrol);
        }
        if (courier.isPresent()) {
            list.add(courier);
        }
        if (transportTrader.isPresent()) {
            list.add(transportTrader);
        }
        if (concurrent.isPresent()) {
            list.add(concurrent);
        }
        if (buyer.isPresent()) {
            list.add(buyer);
        }
        if (pirateHunter.isPresent()) {
            list.add(pirateHunter);
        }
        if (warehouseTenant.isPresent()) {
            list.add(warehouseTenant);
        }
        if (smuggler.isPresent()) {
            list.add(smuggler);
        }
        if (treasureMap.isPresent()) {
            list.add(treasureMap);
        }
        return list;
    }

    /**
     * Retrieve all people who are present
     * @return list of TavernPersents that are absent.
     */
    List<ITavernPerson> asListAllAbsent() {
        List<ITavernPerson> list = new ArrayList<>();
        // nothing added for contract brooker
        // nothing added for sailors
        // nothing added for captain
        // nothing added for weapons dealer
        // nothing added for pirate
        if (!informant.isPresent()) {
            list.add(informant);
        }
        if (!trader.isPresent()) {
            list.add(trader);
        }
        if (!traveler.isPresent()) {
            list.add(traveler);
        }
        // side room
        if (!thieve.isPresent()) {
            list.add(thieve);
        }
        if (!escorte.isPresent()) {
            list.add(escorte);
        }
        if (!fugative.isPresent()) {
            list.add(fugative);
        }
        if (!patrol.isPresent()) {
            list.add(patrol);
        }
        if (!courier.isPresent()) {
            list.add(courier);
        }
        if (!transportTrader.isPresent()) {
            list.add(transportTrader);
        }
        if (!concurrent.isPresent()) {
            list.add(concurrent);
        }
        if (!buyer.isPresent()) {
            list.add(buyer);
        }
        if (!pirateHunter.isPresent()) {
            list.add(pirateHunter);
        }
        if (!warehouseTenant.isPresent()) {
            list.add(warehouseTenant);
        }
        if (!smuggler.isPresent()) {
            list.add(smuggler);
        }
        if (!treasureMap.isPresent()) {
            list.add(treasureMap);
        }
        return list;
	}

    public MapProperty<IPerson, IPlayer> talkingToProperty() {
        return talkingTo;
    }


    public boolean isTalkingToOtherPlayer(ITavernPerson person, IPlayer player) {
        if (talkingTo.get().containsKey(person)) {
            return !player.equals(talkingTo.get().get(person));
        }
        return false; // is talking to no one.
    }

    // Inner classes

    private class PresenceChangeListener implements ChangeListener<Boolean> {
        private final ITavernPerson tavernPerson;

        private PresenceChangeListener(ITavernPerson tavernPerson) {
            this.tavernPerson = tavernPerson;
        }

        @Override
        public void changed(ObservableValue<? extends Boolean> observableValue,
                            Boolean oldValue, Boolean newValue) {
            if (newValue) {
                absentPersons.remove(tavernPerson);
                presentPersons.add(tavernPerson);
            } else {
                presentPersons.remove(tavernPerson);
                absentPersons.addAll(tavernPerson);
            }
        }
    }

    /**
     * Concrete class for an object binding to avoid issues with serialization.
     */
    private class SideRoomBinding extends ObjectBinding<List<ISideRoomPerson>> {
           SideRoomBinding() {
               for (ISideRoomPerson sideRoomPerson : sideRoomPersons) {
                   super.bind(sideRoomPerson.isPresentProperty());
               }
           }
        @Override
        protected List<ISideRoomPerson> computeValue() {
            List<ISideRoomPerson> persons = newArrayList();
            for (ISideRoomPerson sideRoomPerson : sideRoomPersons) {
                if (sideRoomPerson.isPresent()){
                    persons.add(sideRoomPerson);
                }
            }
            return persons;
        }
    }
}
