package ch.sahits.game.openpatrician.model.event;

import ch.sahits.game.event.data.ClockTick;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * List of tasks that are to be executed at a certain time.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 28, 2013
 *
 */
// This class is not annotated as service/component to be able to have different lists under different names
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
public class TimedUpdatableTaskList  {
	@XStreamOmitField
	private static final Logger LOGGER = LoggerFactory.getLogger("EVENTS");
	@XStreamOmitField
	private final Logger log = LoggerFactory.getLogger(getClass());
	@Autowired
	private Date date;
	@Autowired
	@Qualifier("timerEventBus")
	@XStreamOmitField
	private AsyncEventBus timerEventBus;
	@ListType(TimedTask.class)
	private final List<TimedTask> tasks;

	/**
	 * Register as Clock update listener.
	 */
	@PostConstruct
	public void registerWithClock() {
		timerEventBus.register(this);
	}
	@PreDestroy
	private void unregister() {
		timerEventBus.unregister(this);
	}
	public TimedUpdatableTaskList(){
		tasks = new ArrayList<>();
	}
	/**
	 * Add a new task.
	 * @param task to be added
	 */
	public void add(TimedTask task) {
		synchronized (tasks) {
			tasks.add(task);
			LOGGER.info("Add task {}:{}", task.getClass().getName(), task);
		}
	}
	/**
	 * Cancel a timed task prematurely.
	 * @param task to be removed
	 */
	public void remove(TimedTask task) {
		synchronized (tasks) {
			tasks.remove(task);
		}
	}
    @Subscribe
	public void handleClockTick(ClockTick event) {
		LocalDateTime now = date.getCurrentDate();
		synchronized (tasks) {
			List<TimedTask> unmodifieablyCopy = new ArrayList<>(tasks);
			unmodifieablyCopy.stream()
					.filter(task -> task.getExecutionTime().isBefore(now))
					.forEach(task -> {
						try {
							tasks.remove(task);
							task.run();
							LOGGER.info("Executed task {}:{}", task.getClass().getName(), task);
						} catch (Exception e) {
							log.error("Failed to execute timed task", e);
						}
					});
		}
	}

	/**
	 * Retrieve the first task. Only used for tests.
	 * @return first task in the list.
	 */
	TimedTask getFirst() {
		return tasks.get(0);
	}
}
