package ch.sahits.game.openpatrician.model.map.impl;

import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IFutureTowns;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.model.map.ILandBridge;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.map.ITradeRoute;
import ch.sahits.game.openpatrician.model.people.impl.CaptainsState;
import ch.sahits.game.openpatrician.model.sea.PirateNest;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.MultimapType;
import ch.sahits.game.openpatrician.utilities.annotation.SetType;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.geometry.Dimension2D;
import javafx.geometry.Point2D;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Implementation of the game map.
 * This singleton is instaniated by the Game Factory.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 16, 2011
 *
 */
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
public class GameMap implements IMap {
    @ListType(ICity.class)
	private final LinkedList<ICity> cities = new LinkedList<>();
    @Autowired
    @Setter
    private CaptainsState captainsState;
    @Getter
    private Dimension2D dimension;
    @Getter
    private String mapImagePath;
    @Getter
    private String mapBWImagePath;
    @Getter
    private double numberOfPixelPerKilometer;
    @Getter
    @ListType(IFutureTowns.class)
    private final List<IFutureTowns> futureTowns = new ArrayList<>();
    @Getter
    @ListType(ILandBridge.class)
    private final List<ILandBridge> landbridges = new ArrayList<>();
    @Getter
    @ListType(PirateNest.class)
    private final List<PirateNest> pirateNests = new ArrayList<>();
    @MultimapType(key=IPlayer.class,value = ICity.class)
    @Getter
    private final Multimap<IPlayer, ICity> townsInFounding = ArrayListMultimap.create();
    @Getter
    @ListType(Point2D.class)
    private List<Point2D> cityCoordinates = new ArrayList<>();
    @Getter
    @SetType(ITradeRoute.class)
    private Set<ITradeRoute> tradeRoutes = new HashSet<>();
    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    /**
     * Initialize the map by means of a list of its cities.
     * @param cities List of cities for which to set up the map
     * @param dim dimensions of the map
     * @param bwImageName black and white image filename distinguishing land and sea
     * @param mapName image filename
     * @param numberOfPixelPerKilometer ratio of pixels that make up one km on the map
     */
    public void setup(Collection<ICity> cities, Dimension2D dim, String mapName, String bwImageName, double numberOfPixelPerKilometer) {
        dimension = dim;
        mapImagePath = mapName;
        mapBWImagePath = bwImageName;
        this.numberOfPixelPerKilometer = numberOfPixelPerKilometer;
        for (ICity city : cities) {
            add(city);
            cityCoordinates.add(city.getCoordinates());
        }
    }
	@Override
	public int getNumberCities() {
		return cities.size();
	}

	@Override
	public List<ICity> getCities() {
		return Collections.unmodifiableList(cities);
	}

    @Override
    public ICity findCity(String cityName) {
        for (ICity city : cities) {
            if (city.getUniqueID().equals(cityName)) {
                return city;
            }
        }
        return null;
    }


    public void add(ICity city) {
        this.cities.add(city);
    }

    @Override
    public List<ICity> getCities(IPlayer player) {
        ArrayList<ICity> towns = new ArrayList<>(cities);
        towns.addAll(townsInFounding.get(player));
        return Collections.unmodifiableList(towns);
    }

    @Override
    public Optional<ICity> findCity(Point2D location) {
        for (ICity city : getCities()) {
            if (city.getCoordinates().equals(location)) {
              return Optional.of(city);
            }
        }
        return Optional.empty();
    }
    public void addTradeRoute(ITradeRoute tradeRoute) {
        tradeRoutes.add(tradeRoute);
    }
}
