package ch.sahits.game.openpatrician.model.sea;

import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;

/**
 * Hold the state of all current blockades
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Apr 14, 2016
 */
@LazySingleton
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
public class BlockadeState {
    @MapType(key=ICity.class, value = IBlockade.class)
    private Map<ICity, IBlockade> blockadedCity = new HashMap<>();

    /**
     * Add a blockade for the city.
     * @param city which is blockaded
     * @param blockade the blockade metadata
     */
    public void addBlockade(ICity city, IBlockade blockade) {
        blockadedCity.put(city, blockade);
    }

    /**
     * Retrieve the blockade for the city. If the city is not
     * blockaded an Optional.empty is returned.
      * @param city for which the blockade objecd is to be retrieved.
     * @return Optional of the blockade metadata for the city. If there is no blockade, empty will be returned.
     */
    public Optional<IBlockade> getBlockade(ICity city) {
        if (blockadedCity.containsKey(city)) {
            return Optional.of(blockadedCity.get(city));
        } else {
            return Optional.empty();
        }
    }

    public Set<Entry<ICity, IBlockade>> entrySet() {
        return blockadedCity.entrySet();
    }

    /**
     * Finish the blockade.
     * @param city for which the blockade finished
     */
    public void finishBlockade(ICity city) {
        blockadedCity.remove(city);
    }
}
