package ch.sahits.game.openpatrician.sound.impl;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.sound.IPlayableTrack;
import ch.sahits.game.openpatrician.sound.ITrackLoader;
import ch.sahits.game.sound.data.Track;
import ch.sahits.game.sound.data.Tracks;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.oxm.Unmarshaller;

import javax.annotation.PostConstruct;
import javax.xml.transform.stream.StreamSource;
import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.concurrent.ScheduledExecutorService;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jun 17, 2015
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class TrackLoader implements ITrackLoader {
    private final Logger logger = LogManager.getLogger(getClass());

    private final static String SOUND_DIR = "/sound/";
    @MapType(key = String.class, value = IPlayableTrack.class)
    private HashMap<String, IPlayableTrack> soundMap = new HashMap<>();

    @Autowired()
    @Qualifier("jaxb2SoundMarshaller")
    private Unmarshaller unmarshaller;
    @Autowired
    @Qualifier("uiTimer")
    private ScheduledExecutorService uiTimer;

    private final String fileName;

    public TrackLoader(String fileName) {
        this.fileName = fileName;
    }
    @PostConstruct
    private void loadTracks() {
        final StreamSource sourceFromFile = getSourceFromFile(fileName);
        try {
            Tracks tracks = (Tracks) unmarshaller.unmarshal(sourceFromFile);
            for (Track track : tracks.getTrack()) {
                if (track.getType().equals("l")) {
                    loadLoop(track);
                } else if (track.getType().equals("t")) {
                    loadTrack(track);
                } else {
                    logger.warn("Unknown track type "+track.getType()+" for track "+track.getSrc());
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void loadTrack(Track track) {
        URL url = getClass().getResource(SOUND_DIR + track.getSrc());
        MediaTrack media = new MediaTrack(track.getLength(), url.toExternalForm(), uiTimer);
        String trackId = getPrefix(track.getSrc());
        soundMap.put(trackId, media);
    }

    private void loadLoop(Track track) {
        URL url = getClass().getResource(SOUND_DIR + track.getSrc());
        LoopTrack loop = new LoopTrack(track.getLength(), url.toExternalForm());
        String trackId = getPrefix(track.getSrc());
        soundMap.put(trackId, loop);
    }

    @Override
    public IPlayableTrack loadTrack(String trackID) {
        return soundMap.get(trackID);
    }
    private StreamSource getSourceFromFile(String fileName) {
        String imsFNm = "/" + fileName;
        logger.info("Reading file: " + imsFNm);
        return new StreamSource(getClass().getResourceAsStream(imsFNm));
    }
    /**
     * Extract the name before the last '.'
     * @param fnm
     * @return
     */
    private String getPrefix(String fnm)
    // extract name before '.' of filename
    {
        int posn;
        if ((posn = fnm.lastIndexOf(".")) == -1) {
            logger.debug("No prefix found for filename: " + fnm);
            return fnm;
        } else
            return fnm.substring(0, posn);
    } // end of getPrefix()
}
