package ch.sahits.game.openpatrician.standalone;

import ch.sahits.game.openpatrician.utilities.LocalStorage;
import ch.sahits.game.openpatrician.utilities.plugin.BasePluginDefinition;
import ch.sahits.game.openpatrician.utilities.plugin.PluginConfiguration;
import ch.sahits.game.openpatrician.utilities.service.UnzipUtility;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.RollingFileAppender;
import org.springframework.stereotype.Service;

import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import java.io.File;
import java.io.IOException;
import java.util.Properties;

/**
 * Various service required at application start.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Mar 23, 2018
 */
@Service
@Slf4j
public class StartupService {

    /**
     * Check the minimum version requirement for the application.
     * @return true if the Java version is supported.
     */
    public boolean checkVersion() {
        String jreVersion = (String) System.getProperties().get("java.version");
        if (!jreVersion.startsWith("9.0")) {
            log.error("JRE must be of version 9");
            System.out.println("JRE must be of version 9");
            return false;
        }
        return true;
    }

    /**
     * Log some basic evironment information.
     */
    public void logEnvironment() {
        Properties sysprops = System.getProperties();
        LoggerContext logContext = (LoggerContext) LogManager.getContext();
        if (log.isInfoEnabled()) {
            log.info("Logging into file: "+((RollingFileAppender)logContext.getConfiguration().getAppender("FileAppender")).getFileName());
        }
        StringBuilder sb = new StringBuilder();
        sb.append("Java VM version").append(": ").append(sysprops.get("java.vm.version"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("Java runtime version").append(": ").append(sysprops.get("java.runtime.version"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("Java VM vendor").append(": ").append(sysprops.get("java.vm.vendor"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("Java VM name").append(": ").append(sysprops.get("java.vm.name"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("Java version").append(": ").append(sysprops.get("java.version"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("Java specification version").append(": ").append(sysprops.get("java.specification.version"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("Graphics environment").append(": ").append(sysprops.get("java.awt.graphicsenv"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("User country").append(": ").append(sysprops.get("user.country"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("User language").append(": ").append(sysprops.get("user.language"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("OS architecture").append(": ").append(sysprops.get("os.arch"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("OS name").append(": ").append(sysprops.get("os.name"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("OS version").append(": ").append(sysprops.get("os.version"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("Classpath").append(": ").append(sysprops.get("java.class.path"));
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
        sb = new StringBuilder();
        sb.append("Start command").append(": ").append(sysprops.get("sun.java.command")); // ?? is this generally availalbe
        if (log.isInfoEnabled()) {
            log.info(sb.toString());
        }
    }

    /**
     * Unpack any plagins.
     */
    public void unpackPlugins() {
        LocalStorage localeStorage = new LocalStorage();
        File pluginDir = localeStorage.getPluginDirectory();
        String[] zipFiles = pluginDir.list((dir, name) -> name.endsWith(".zip"));
        UnzipUtility unzipper = new UnzipUtility();
        for (String zipFile : zipFiles) {

            String zipFileName = localeStorage.getPluginDirectoryPath()+File.separatorChar+zipFile;
            try {
                String pluginFilePath = unzipper.unzipPluginDescriptor(zipFileName);
                File pluginDesc = new File(pluginFilePath);
                PluginConfiguration pluginConfiguration = new PluginConfiguration();
                Unmarshaller um = pluginConfiguration.jaxbPluginUnMarshaller();
                BasePluginDefinition plugin = (BasePluginDefinition) um.unmarshal(pluginDesc);

                String outputdirectory = localeStorage.getPluginTypeDirecotryPath(plugin.getType(), zipFile);
                // If the outputdirectory exists delete it.
                if (isPluginUnziped(outputdirectory)) {
                    System.out.println("Should delete old plugin");
                    File dir = new File(outputdirectory);
                    FileUtils.deleteDirectory(dir);
                }
                unzipper.unzip(zipFileName, outputdirectory);
                File f = new File(zipFileName);
                f.delete();
            } catch (IOException e) {
                log.warn("Failed to extract plugin "+zipFile, e);
            } catch (JAXBException e) {
                e.printStackTrace();
            }
        }
    }

    private boolean isPluginUnziped(String pluginPath) {
        File f = new File(pluginPath);
        return f.exists();
    }
}
