package ch.sahits.game.openpatrician.standalone;

import ch.sahits.game.openpatrician.display.CheatKeyEventListener;
import ch.sahits.game.openpatrician.display.CommandLineArguments;
import ch.sahits.game.openpatrician.display.IOpenPatricianApplicationWindow;
import ch.sahits.game.openpatrician.display.SceneChangeable;
import ch.sahits.game.openpatrician.display.StandaloneConfiguration;
import ch.sahits.game.openpatrician.display.javafx.MainGameView;
import ch.sahits.game.openpatrician.display.scene.BaseStartupScene;
import ch.sahits.game.openpatrician.display.service.UIFactory;
import ch.sahits.game.openpatrician.event.EGameStatusChange;
import ch.sahits.game.openpatrician.event.GameStateChange;
import ch.sahits.game.openpatrician.javafx.OpenPatricianScene;
import ch.sahits.game.openpatrician.javafx.event.AggregateEventHandler;
import ch.sahits.game.openpatrician.javafx.event.CloseApplicationEventHandler;
import ch.sahits.game.openpatrician.javafx.event.MouseClickLocationEventHandler;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.service.GamePropertyUtility;
import ch.sahits.game.openpatrician.utilities.spring.DependentAnnotationConfigApplicationContext;
import com.google.common.eventbus.AsyncEventBus;
import javafx.application.Application;
import javafx.event.EventHandler;
import javafx.scene.ImageCursor;
import javafx.scene.Scene;
import javafx.scene.image.Image;
import javafx.scene.input.KeyCode;
import javafx.scene.input.KeyEvent;
import javafx.stage.Stage;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Options;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.builder.SpringApplicationBuilder;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Import;
import org.springframework.context.support.AbstractApplicationContext;

import java.io.IOException;
import java.net.URL;

/**
 * Spring boot application starting the JavaFX UI.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Mar 23, 2018
 */
@Slf4j
@SpringBootApplication
@Import(StandaloneConfiguration.class)
@ClassCategory(EClassCategory.STARTUP)
public class OpenPatricianApplication extends Application implements IOpenPatricianApplicationWindow {

    private StartupService startupService;
    private GamePropertyUtility gamePropertyUtility;

    private int width;
    private int height;
    private boolean fullscreen;
    private Stage primaryStage;

    private final AggregateEventHandler<KeyEvent> keyEventHandlerAggregate;
    private final MouseClickLocationEventHandler mouseClickEventHandler;

    private ApplicationContext context;

    public OpenPatricianApplication() {
        width = MIN_WIDTH;
        height = MIN_HEIGHT;
        this.fullscreen = false;
        keyEventHandlerAggregate = new AggregateEventHandler<>();

        CloseApplicationEventHandler closeEventHandler = new CloseApplicationEventHandler();
        mouseClickEventHandler = new MouseClickLocationEventHandler();
        EventHandler<KeyEvent> fullScreenEventHandler = event -> {
            try {
                if (event.getCode().equals(KeyCode.F) && event.isControlDown()) {
                    updateFullscreenMode();
                }
            } catch (RuntimeException e) {
                log.error("Failed to switch to/from fullscreen mode", e);
            }
        };
        EventHandler<KeyEvent> closeEventWindowKeyHandler = event -> {
            if (event.getCode().equals(KeyCode.ESCAPE)) {
                log.info("Pressed ESC");
                context.getBean(MainGameView.class).closeEventView();
            }
        };
        addKeyEventHandler(closeEventHandler);
        addKeyEventHandler(fullScreenEventHandler);
        addKeyEventHandler(closeEventWindowKeyHandler);
    }

    /**
     * Add a key event handler to the application.
     * @param eventHandler to be added.
     */
    private void addKeyEventHandler(EventHandler<KeyEvent> eventHandler) {
        keyEventHandlerAggregate.addEventHandler(eventHandler);
    }

    public static void main(String[] args) {
        launch(args);
    }

    @Override
    public void init() {
        SpringApplicationBuilder builder = new SpringApplicationBuilder(OpenPatricianApplication.class);
        context = builder.contextClass(DependentAnnotationConfigApplicationContext.class).run(getParameters().getRaw().toArray(new String[0]));
        this.startupService = context.getBean(StartupService.class);
        this.gamePropertyUtility = context.getBean(GamePropertyUtility.class);
        if (startupService.checkVersion()) {
            startupService.logEnvironment();

            CommandLineArguments cmdHelper = new CommandLineArguments();
            Options opts = cmdHelper.createCommandLineOptions();
            CommandLine cmdLine = cmdHelper.parseCommandLine(opts, getParameters().getRaw().toArray(new String[getParameters().getRaw().size()]));
            if (cmdLine.hasOption(CommandLineArguments.HELP_OPTION)){
                cmdHelper.printHelp(opts);
                System.exit(0);
            }
            if (cmdLine.hasOption(CommandLineArguments.VERSION_OPTION)) {
                System.out.println("OpenPatrician version: "+OpenPatricianApplication.class.getPackage().getImplementationVersion());
                System.exit(0);
            }
            cmdHelper.persistAsPropertyFile(cmdLine);
        }
    }

    @Override
    public void start(Stage primaryStage) {
        this.primaryStage = primaryStage;
        this.primaryStage.setMinWidth(MIN_WIDTH);
        this.primaryStage.setMinHeight(MIN_HEIGHT);
        primaryStage.getIcons().add(new Image(getClass().getResourceAsStream("/icons/trade-icon.png")));
        UIFactory uiFactory = context.getBean(UIFactory.class);
        uiFactory.setApplicationWindow(this);
        BaseStartupScene startupS = uiFactory.getStartupScene();
        Scene defaultScene = new Scene(startupS.getRoot(), width, height);
        defaultScene.getStylesheets().add("/styles/font.css");

        this.fullscreen = Boolean.valueOf((String) gamePropertyUtility.getProperties().get("window.fullscreen"));
        startupS.setSceneChangeable(this);
        defaultScene.setOnMousePressed(mouseClickEventHandler);
        defaultScene.setOnKeyPressed(keyEventHandlerAggregate);
        try {
            CheatKeyEventListener cheatListener = context.getBean(CheatKeyEventListener.class);
            if (cheatListener != null) {
                addKeyEventHandler(cheatListener);
            }
        } catch (Exception e) {
            // the cheat listener is no defined for the context.
            e.printStackTrace();
        }

        setCursor(defaultScene);

        primaryStage.setFullScreen(fullscreen);
        primaryStage.setFullScreenExitHint("");
        primaryStage.setTitle("OpenPatrician");
        primaryStage.setScene(defaultScene);
        primaryStage.show();
        // Compilation error with Maven
//		if (System.getProperties().containsKey("showScenicView")) {
//			ScenicView.show(defaultScene);
//		}
    }

    private void setCursor(Scene scene) {
        URL url = getClass().getResource("/icons/64/cursor.png");
        try {
            Image img = new Image(url.openStream());
            scene.setCursor(new ImageCursor(img));
        } catch (IOException e) {
            log.warn("Failed to load cursor icon from {}", url);
        }
    }

    /**
     * @see SceneChangeable#changeScene(OpenPatricianScene)
     */
    @Override
    public void changeScene(final OpenPatricianScene scene) {
        primaryStage.getScene().setOnMousePressed(mouseClickEventHandler);
        primaryStage.getScene().setOnKeyPressed(keyEventHandlerAggregate);

        primaryStage.getScene().setRoot(scene.getRoot());
    }
    /**
     * Toggle between full screen and non full screen mode.
     */
    public void updateFullscreenMode() {
        fullscreen = !fullscreen;
        primaryStage.setFullScreen(fullscreen);
    }

    @Override
    public double getSceneWidth() {
        return primaryStage.getScene().getWidth();
    }

    @Override
    public double getSceneHeight() {
        return primaryStage.getScene().getHeight();
    }

    @Override
    public void stop() throws Exception {
        System.out.println("Stopping the UI Application");

        stopUIApplicationContext();
        super.stop();
    }

    /**
     * Closing the application context for the user interface.
     */
    private void stopUIApplicationContext() {
        AsyncEventBus eventBus = (AsyncEventBus) context.getBean("serverClientEventBus");
        eventBus.post(new GameStateChange(EGameStatusChange.SHUTDOWN));
        ((AbstractApplicationContext)context).close();
    }
}
