/*
 * Copyright 2006 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License")
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ch.sharedvd.jaxbPlugins.nonNullGetter;

import com.sun.codemodel.*;
import com.sun.tools.xjc.Options;
import com.sun.tools.xjc.Plugin;
import com.sun.tools.xjc.model.Aspect;
import com.sun.tools.xjc.model.CClassInfo;
import com.sun.tools.xjc.outline.ClassOutline;
import com.sun.tools.xjc.outline.Outline;
import org.xml.sax.ErrorHandler;

import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;

/**
 * Support a non null getter in addition to the default (JavaBean) setter methods.<br>
 * <p>
 * The initial idea is simply to add a "with*" method to the generated class
 * for every "set*" method encountered,
 * with the only functional difference of returning the class instance, instead of void.
 * <p>
 * <strong>Enhancement on 11 June 2006:</strong><br>
 * Provide fluent setter api for Lists, with support of variable arguments.
 * <p>
 * This enhancement was suggested by Kenny MacLeod &lt;kennym@kizoom.com&gt;,
 * and endorsed by Kohsuke Kawaguchi &lt;Kohsuke.Kawaguchi@sun.com&gt;.
 * Here is quoted from the original request:
 * <p>
 * By default, XJC represents Lists by generating a getter method, but no setter.
 * This is impossible to chain with fluent-api.
 * How about the plugin generates a withXYZ() method for List properties,
 * taking as it's parameters a vararg list.  For example:
 * <blockquote><pre>
 * // This method is generated by vanilla XJC
 * public List&lt;OtherType&gt; getMyList() {
 *   if (myList == null) {
 *     myList = new ArrayList&lt;OtherType&gt;();
 *   }
 *   return myList;
 * }
 *
 * // This would be generated by fluent-api
 * public MyClass withMyList(OtherType... values) {
 *   for(OtherType value : values) {
 *     getMyList().add(value);
 *   }
 *   return this;
 * }
 * </pre></blockquote>
 *
 * @author Hanson Char
 */
public class XjcNonNullGetterPlugin extends Plugin {
    @Override
    public String getOptionName() {
        return "XNonNullGetter-api";
    }

    @Override
    public String getUsage() {
        return "  -XNonNullGetter-api        :  non null getter api for generated code";
    }

    @Override
    public boolean run(Outline aOutline, Options opt, ErrorHandler errorHandler) {
        // Process every pojo class generated by jaxb
        for (ClassOutline classOutline : aOutline.getClasses()) {
            CClassInfo classInfo = classOutline.target;
            treatOneClass(aOutline, classInfo);
        }
        return true;
    }

    /**
     * @param aOutline
     * @param aClassInfo
     */
    private void treatOneClass(Outline aOutline, CClassInfo aClassInfo) {

        JClass implClass = aClassInfo.toType(aOutline, Aspect.IMPLEMENTATION);
        if (null == implClass) {
            throw new NullPointerException(
                    "Classe d'implémentation inconnue pour " + aClassInfo.getName());
        }
        if (!(implClass instanceof JDefinedClass)) {
            throw new NullPointerException(
                    "N'est pas une JDefinedClass " + aClassInfo.getName());
        }
        treatOneClass(aOutline, (JDefinedClass) implClass);
    }

    private void treatOneClass(Outline aOutline, JDefinedClass aClass) {

        for (Iterator<Entry<String, JFieldVar>> it = aClass.fields().entrySet().iterator(); it
                .hasNext(); ) {
            Entry<String, JFieldVar> entry = (Entry<String, JFieldVar>) it.next();
            String fieldName = entry.getKey();
            JFieldVar field = entry.getValue();

            JType type = field.type();

            boolean isList = false;
            if (type instanceof JClass) {
                JClass jclass = JClass.class.cast(type);
                List<JClass> typeParams = jclass.getTypeParameters();
                if (typeParams.size() == 1) {
                    jclass = typeParams.get(0);
                    isList = true;
                }
                boolean isGeneratedType = false;
                if ((jclass instanceof JDefinedClass)
                        && (ClassType.ENUM != ((JDefinedClass) jclass).getClassType())) {
                    JClass outerClass = jclass;
                    while (!isGeneratedType && (null != outerClass)) {
                        String typeFullName = outerClass.fullName();
                        isGeneratedType = (null != aOutline.getCodeModel()._getClass(typeFullName));
                        outerClass = outerClass.outer();
                    }
                    if (isGeneratedType) {
                        if (isList) {
                            treatListField(aClass, fieldName, field, jclass);
                        } else {
                            treatSimpleField(aClass, fieldName, field);
                        }
                    }
                }
            }
        }
    }

    private void treatSimpleField(JDefinedClass aImplClass, String aFieldName, JFieldVar aField) {
        JMethod nonNullGetterMethod = aImplClass.method(JMod.PUBLIC, aField.type(), aFieldName);
        JBlock jblock = nonNullGetterMethod.body();

        jblock._if(JExpr._null().eq(JExpr.refthis(aFieldName)))
                ._then().assign(JExpr.refthis(aFieldName), JExpr._new(aField.type()));
        jblock._return(JExpr.refthis(aFieldName));
    }

    private void treatListField(JDefinedClass aImplClass, String aFieldName, JFieldVar aField,
                                JType aType) {

        StringBuffer getterMethodName = new StringBuffer("get");
        getterMethodName.append(aFieldName.substring(0, 1).toUpperCase());
        getterMethodName.append(aFieldName.substring(1));
//        JMethod getterMethod = aImplClass.getMethod(getterMethodName.toString(), new JType[] {});

        JMethod nonNullGetterMethod = aImplClass.method(JMod.PUBLIC, aType, aFieldName);
        JBlock jblock = nonNullGetterMethod.body();

        JVar newEl = jblock.decl(aType, "newElement", JExpr._new(aType));
        jblock.directStatement(getterMethodName + "().add(newElement);");
        jblock._return(newEl);
    }
}
