package ch.squaredesk.nova.metrics;

import ch.squaredesk.nova.tuples.Pair;
import ch.squaredesk.nova.tuples.Tuple3;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.dropwizard.metrics5.Metric;
import io.dropwizard.metrics5.MetricName;

import java.util.HashMap;
import java.util.Map;

/**
 * This class takes a dump, generated by Nova's Metrics instance and converts them into a map
 * that can easily be serialized and deserialized.
 *
 * The resulting map contains one entry per metric in the dump, each can be looked up by the Metric name. Additionally,
 * we add the following
 *   - timestamp
 *   - hostName
 *   - hostAddress
 *
 * Every Metric itself is again represented as a Map, one MapEntry per attribute. Additionally, we add
 *   - type
 *
 */
public class MetricsConverter {
    private static final ObjectMapper objectMapper = new ObjectMapper();

    private MetricsConverter() {
    }

    public static Map<String, Map<String, Object>> convert(Map<MetricName, Metric> dump) {
        return convert(dump,null);
    }

    public static Map<String, Map<String, Object>> convert(Map<MetricName, Metric> dump, Map<String, Object> additionalAttributes) {
        HashMap<String, Map<String, Object>> returnValue = new HashMap<>();

        dump.entrySet().stream()
                // for each metric, create a tuple containing the type, the name and the metric itself
                .map(entry -> new Tuple3<>(
                        entry.getValue().getClass().getSimpleName(),
                        entry.getKey(),
                        entry.getValue()))
                // convert each metric to a Map and enrich with our default values, as well as the
                // (eventually) passed additionalAttributes
                .map(tupleTypeAndNameAndMetric -> {
                    Map<String, Object> map = toMap(tupleTypeAndNameAndMetric._3);
                    map.put("type", tupleTypeAndNameAndMetric._1);
                    if (additionalAttributes != null) {
                        map.putAll(additionalAttributes);
                    }
                    return new Pair<>(tupleTypeAndNameAndMetric._2, map);
                })
                // and add it to the return value
                .forEach(metricNameMapPair -> returnValue.put(metricNameMapPair._1.toString(), metricNameMapPair._2));

        return returnValue;
    }

    private static Map<String, Object> toMap (CompoundMetric compoundMetric) {
        Map<String, Object> returnValue = new HashMap<>();
        Map<MetricName, Object> values = compoundMetric.getValues();

        if (!values.isEmpty()) {
            values.forEach((key, value) -> returnValue.put(key.toString(), value));
        }
        returnValue.put("type", compoundMetric.getClass().getSimpleName());

        return returnValue;
    }

    private static Map<String, Object> toMap(Metric metric) {
        if (metric instanceof CompoundMetric) {
            return toMap((CompoundMetric) metric);
        } else {
            Map map = objectMapper.convertValue(metric, Map.class);
            map.put("type", metric.getClass().getSimpleName());
            return map;
        }
    }

}
